<?xml version="1.0" encoding="UTF-8"?>
<!--
******************************************************************************************************************

	Transformation from PeppolBIS-2 to OIOUBL 3

	Publisher:         Erhvervsstyrelsen
	Created date:      2024-09-30
	Last changed date: 

	Description:       Transform PeppolBIS Invoice or Credit note to a valid OIOUBL 3  
	Rights:            It can be used following the Common Creative License

	Stylesheet Version: 3.2.1.22ca3408

	For more information contact Erhvervsstyrelsen

*****************************************************************************************************************

-->
<xsl:stylesheet version="2.0"
    xmlns:cac="urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2"
    xmlns:cbc="urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2"
    xmlns:ccts="urn:oasis:names:specification:ubl:schema:xsd:CoreComponentParameters-2"
    xmlns:cec="urn:oasis:names:specification:ubl:schema:xsd:CommonExtensionComponents-2"
    xmlns:sdt="urn:oasis:names:specification:ubl:schema:xsd:SpecializedDatatypes-2"
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    xmlns:ext="urn:oasis:names:specification:ubl:schema:xsd:CommonExtensionComponents-2"
    xmlns:udt="urn:oasis:names:specification:ubl:schema:xsd:UnqualifiedDataTypes-2"
    xmlns:qdt="urn:oasis:names:specification:ubl:schema:xsd:QualifiedDataTypes-2"
    xmlns:xs="http://www.w3.org/2001/XMLSchema" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

    <xsl:output method="xml" encoding="UTF-8" indent="yes"/>

    <!--  ********************
    */
    */  Global values
    */
    **************************  -->
    <xsl:variable name="rootName" select="name(/*)"/>

    <xsl:variable name="root" select="/*"/>


    <!--  ********************
    */
    */  Start: Identify documenttype (Invoice or CreditNote)
    */
    **************************  -->
    <xsl:template match="/">

        <xsl:choose>

            <xsl:when test="$rootName = 'Invoice'">

                <xsl:apply-templates mode="Invoice"/>

            </xsl:when>

            <xsl:when test="$rootName = 'CreditNote'">

                <xsl:apply-templates mode="CreditNote"/>

            </xsl:when>

            <xsl:otherwise>

                <xsl:text>Fatal error: Unsupported documenttype! This stylesheet only supports conversion of a Peppol BIS 2.0 Invoice or CreditNote.</xsl:text>

            </xsl:otherwise>

        </xsl:choose>
    </xsl:template>


    <!--  ********************
    */
    */ Create the main structure for the document type and process the transformation
    */
    **************************  -->
    <!-- Invoice -->

    <xsl:template match="*[local-name() = 'Invoice']" mode="Invoice">

        <Invoice xmlns="urn:oasis:names:specification:ubl:schema:xsd:Invoice-2">

            <!-- Transform the invoice header information -->
            <xsl:call-template name="CreateDocumentHeader"/>

            <!-- Transform the invoice line information -->
            <xsl:apply-templates select="cac:InvoiceLine"/>

        </Invoice>
    </xsl:template>

    <!-- Create InvoiceLine elemnent per line -->
    <xsl:template match="cac:InvoiceLine">

        <cac:InvoiceLine>
            <xsl:call-template name="createDocumentLines"/>
        </cac:InvoiceLine>

    </xsl:template>


    <!-- Credit Note -->
    <xsl:template match="*[local-name() = 'CreditNote']" mode="CreditNote">

        <CreditNote xmlns="urn:oasis:names:specification:ubl:schema:xsd:CreditNote-2">

            <!-- Transform the CreditNote header information -->
            <xsl:call-template name="CreateDocumentHeader"/>

            <!-- Transform the CreditNote line information -->
            <xsl:apply-templates select="cac:CreditNoteLine"/>

        </CreditNote>
    </xsl:template>


    <!-- Create CreditNoteLine elemnent per line -->
    <xsl:template match="cac:CreditNoteLine">

        <cac:CreditNoteLine>
            <xsl:call-template name="createDocumentLines"/>
        </cac:CreditNoteLine>

    </xsl:template>


    <!--  ********************
    */
    */ Create the header content for the document type
    */
    **************************  -->

    <xsl:template name="CreateDocumentHeader">

        <cbc:UBLVersionID>2.1</cbc:UBLVersionID>

        <xsl:choose>
            <xsl:when test="$rootName = 'Invoice'">

                <cbc:CustomizationID>urn:fdc:oioubl.dk:trns:billing:invoice:3.0</cbc:CustomizationID>

            </xsl:when>
            <xsl:otherwise>

                <cbc:CustomizationID>urn:fdc:oioubl.dk:trns:billing:creditnote:3.0</cbc:CustomizationID>

            </xsl:otherwise>
        </xsl:choose>

        <cbc:ProfileID>urn:fdc:oioubl.dk:bis:billing_with_response:3</cbc:ProfileID>

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:IssueDate" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:DueDate" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:InvoiceTypeCode" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:CreditNoteTypeCode" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:Note" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:TaxPointDate" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:DocumentCurrencyCode" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:TaxCurrencyCode" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:AccountingCost" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:BuyerReference" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:InvoicePeriod" copy-namespaces="no"/>

        <xsl:copy-of select="cac:OrderReference" copy-namespaces="no"/>

        <xsl:copy-of select="cac:DespatchDocumentReference" copy-namespaces="no"/>

        <xsl:copy-of select="cac:ReceiptDocumentReference" copy-namespaces="no"/>

        <xsl:copy-of select="cac:ContractDocumentReference" copy-namespaces="no"/>

        <xsl:copy-of select="cac:AdditionalDocumentReference" copy-namespaces="no"/>

        <xsl:apply-templates select="cac:AccountingSupplierParty"/>

        <xsl:apply-templates select="cac:AccountingCustomerParty"/>

        <xsl:apply-templates select="cac:PayeeParty"/>

        <xsl:apply-templates select="cac:BuyerCustomerParty"/>

        <xsl:apply-templates select="cac:SellerSupplierParty"/>

        <xsl:apply-templates select="cac:TaxRepresentativeParty"/>

        <xsl:copy-of select="cac:PaymentMeans" copy-namespaces="no"/>

        <xsl:copy-of select="cac:PaymentTerms" copy-namespaces="no"/>

        <xsl:copy-of select="cac:AllowanceCharge" copy-namespaces="no"/>

        <xsl:copy-of select="cac:TaxTotal" copy-namespaces="no"/>

        <xsl:copy-of select="cac:LegalMonetaryTotal" copy-namespaces="no"/>


    </xsl:template>


    <!--  ********************
    */
    */ Create the line content for the document type
    */
    **************************  -->

    <xsl:template name="createDocumentLines">

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:Note" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:InvoicedQuantity" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:CreditedQuantity" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:LineExtensionAmount" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:AccountingCost" copy-namespaces="no"/>

        <xsl:copy-of select="cac:OrderLineReference" copy-namespaces="no"/>

        <xsl:copy-of select="cac:DocumentReference" copy-namespaces="no"/>

        <!-- For CreditNote AllowanceCharge comres before TaxTotal-->
        <xsl:if test="$rootName = 'Invoice'">
            <xsl:copy-of select="cac:AllowanceCharge" copy-namespaces="no"/>
        </xsl:if>


        <xsl:choose>
            <xsl:when test="cac:TaxTotal">
                <xsl:copy-of select="cac:TaxTotal" copy-namespaces="no"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:call-template name="createTaxTotal"/>
            </xsl:otherwise>
        </xsl:choose>


        <!-- For Invoice AllowanceCharge comres after -->
        <xsl:if test="$rootName = 'CreditNote'">
            <xsl:copy-of select="cac:AllowanceCharge" copy-namespaces="no"/>
        </xsl:if>


        <xsl:copy-of select="cac:Item" copy-namespaces="no"/>

        <xsl:copy-of select="cac:Price" copy-namespaces="no"/>

    </xsl:template>



    <!-- 
        Create Party-element for complex Paety
    -->
    <xsl:template
        match="cac:AccountingSupplierParty | cac:AccountingCustomerParty | cac:BuyerCustomerParty | cac:SellerSupplierParty">

        <xsl:element name="{name()}">

            <cac:Party>

                <cbc:EndpointID>

                    <xsl:copy-of select="cac:Party/cbc:EndpointID/@*" copy-namespaces="no"/>
                    <xsl:value-of select="
                            if (cac:Party/cbc:EndpointID/@schemeID = '0184')
                            then
                                replace(cac:Party/cbc:EndpointID, '^DK', '')
                            else
                                cac:Party/cbc:EndpointID"/>
                </cbc:EndpointID>

                <xsl:if test="cac:Party/cac:PartyIdentification">

                    <cac:PartyIdentification>
                        <cbc:ID schemeID="{cac:Party/cac:PartyIdentification/cbc:ID/@schemeID}">

                            <xsl:copy-of select="cac:Party/cac:PartyIdentification/@*"
                                copy-namespaces="no"/>

                            <xsl:value-of select="
                                    if (cac:Party/cac:PartyIdentification/cbc:ID/@schemeID = '0184')
                                    then
                                        replace(cac:Party/cac:PartyIdentification/cbc:ID, '^DK', '')
                                    else
                                        cac:Party/cac:PartyIdentification/cbc:ID"/>

                        </cbc:ID>
                    </cac:PartyIdentification>

                </xsl:if>

                <xsl:copy-of select="cac:Party/cac:PartyName" copy-namespaces="no"/>

                <cac:PostalAddress>
                    <xsl:apply-templates select="cac:Party/cac:PostalAddress"/>
                </cac:PostalAddress>

                <xsl:copy-of select="cac:Party/cac:PartyTaxScheme"/>

                <xsl:apply-templates select="cac:Party/cac:PartyLegalEntity"/>

                <xsl:apply-templates select="cac:Party/cac:Contact"/>

            </cac:Party>

        </xsl:element>
    </xsl:template>


    <!-- 
        Create Party-element for simple Party 
    -->
    <xsl:template match="cac:PayeeParty">

        <xsl:element name="{name()}">


            <xsl:if test="cac:Party/cac:PartyIdentification">

                <cac:PartyIdentification>
                    <cbc:ID schemeID="{cac:Party/cac:PartyIdentification/cbc:ID/@schemeID}">

                        <xsl:copy-of select="cac:Party/cac:PartyIdentification/@*"
                            copy-namespaces="no"/>

                        <xsl:value-of select="
                                if (cac:Party/cac:PartyIdentification/cbc:ID/@schemeID = '0184')
                                then
                                    replace(cac:Party/cac:PartyIdentification/cbc:ID, '^DK', '')
                                else
                                    cac:Party/cac:PartyIdentification/cbc:ID"/>

                    </cbc:ID>
                </cac:PartyIdentification>

            </xsl:if>

            <xsl:copy-of select="cac:PartyName" copy-namespaces="no"/>


            <xsl:apply-templates select="cac:Contact"/>



        </xsl:element>
    </xsl:template>


    <!-- 
        Create Party-element for simple Party 
    -->
    <xsl:template match="cac:TaxRepresentativeParty">

        <xsl:element name="{name()}">

            <xsl:copy-of select="cac:PartyName" copy-namespaces="no"/>

            <cac:PostalAddress>
                <xsl:apply-templates select="cac:PostalAddress"/>
            </cac:PostalAddress>

            <xsl:copy-of select="cac:PartyTaxScheme"/>


        </xsl:element>
    </xsl:template>




    <!-- 
        Ensure BuildingNumber & StreetName
    -->
    <xsl:template match="cac:PostalAddress">


        <xsl:if test="cbc:BuildingNumber and not(cbc:StreetName)">
            <cbc:StreetName>Default</cbc:StreetName>
            <xsl:copy-of select="cbc:AdditionalStreetName"/>
            <xsl:copy-of select="cbc:BuildingNumber"/>
        </xsl:if>



        <xsl:if test="cbc:StreetName and not(cbc:BuildingNumber)">
            <xsl:copy-of select="cbc:StreetName"/>
            <xsl:copy-of select="cbc:AdditionalStreetName"/>
            <cbc:BuildingNumber>Default</cbc:BuildingNumber>
        </xsl:if>

        <xsl:copy-of
            select="cbc:CityName | cbc:PostalZone | cbc:CountrySubentity | cac:AddressLine | cac:Country"/>



    </xsl:template>

    <!--
        Remove DK as prefix in CompanyID is schemeID = 0184
    -->

    <xsl:template match="cac:PartyLegalEntity">

        <cac:PartyLegalEntity>

            <xsl:copy-of select="cbc:RegistrationName" copy-namespaces="no"/>

            <cbc:CompanyID schemeID="{cbc:CompanyID/@schemeID}">

                <xsl:value-of select="
                        if (cbc:CompanyID/@schemeID = '0184' and starts-with(cbc:CompanyID, 'DK')) then
                            substring-after(cbc:CompanyID, 'DK')
                        else
                            cbc:CompanyID"/>
            </cbc:CompanyID>
        </cac:PartyLegalEntity>

    </xsl:template>


    <!--
        Insert 00 as prefix for phone number
    -->

    <xsl:template match="cac:Contact">

        <cac:Contact>

            <xsl:copy-of select="cbc:ID | cbc:Name" copy-namespaces="no"/>

            <xsl:if test="cbc:Telephone">

                <cbc:Telephone>
                    <xsl:value-of select="
                            if (starts-with(cbc:Telephone, '00')) then
                                cbc:Telephone
                            else
                                concat('00', cbc:Telephone)"/>
                </cbc:Telephone>

            </xsl:if>

            <xsl:copy-of select="cbc:ElectronicMail | cbc:Note" copy-namespaces="no"/>


        </cac:Contact>

    </xsl:template>

    <!-- 
        Creates TaxTotal (on line) if missing in the source document - Mandentory in OIOUBL 3 
    -->
    <xsl:template name="createTaxTotal">
        <xsl:param name="path"/>

        <xsl:comment> TaxTotal is created based on ClassifiedTaxCategory and LineExtensionAmount </xsl:comment>

        <cac:TaxTotal>

            <cbc:TaxAmount currencyID="{cbc:LineExtensionAmount/@currencyID}">
                <xsl:choose>
                    <xsl:when test="cac:Item/cac:ClassifiedTaxCategory/cbc:Percent">
                        <xsl:value-of
                            select="format-number(cbc:LineExtensionAmount * (cac:Item/cac:ClassifiedTaxCategory/cbc:Percent div 100), '0.00')"
                        />
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="'0.00'"/>
                    </xsl:otherwise>
                </xsl:choose>
            </cbc:TaxAmount>

            <cac:TaxSubtotal>
                <cbc:TaxableAmount currencyID="{cbc:LineExtensionAmount/@currencyID}">
                    <xsl:value-of select="cbc:LineExtensionAmount"/>
                </cbc:TaxableAmount>

                <cbc:TaxAmount currencyID="{cbc:LineExtensionAmount/@currencyID}">
                    <xsl:choose>
                        <xsl:when test="cac:Item/cac:ClassifiedTaxCategory/cbc:Percent">

                            <xsl:value-of
                                select="format-number(cbc:LineExtensionAmount * (cac:Item/cac:ClassifiedTaxCategory/cbc:Percent div 100), '0.00')"/>

                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="'0.00'"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </cbc:TaxAmount>

                <cac:TaxCategory>
                    <cbc:ID>
                        <xsl:value-of select="cac:Item/cac:ClassifiedTaxCategory/cbc:ID"/>
                    </cbc:ID>

                    <xsl:if test="cac:Item/cac:ClassifiedTaxCategory/cbc:Percent">
                        <cbc:Percent>
                            <xsl:value-of select="cac:Item/cac:ClassifiedTaxCategory/cbc:Percent"/>
                        </cbc:Percent>
                    </xsl:if>

                    <cac:TaxScheme>
                        <cbc:ID>
                            <xsl:value-of
                                select="cac:Item/cac:ClassifiedTaxCategory/cac:TaxScheme/cbc:ID"/>
                        </cbc:ID>
                    </cac:TaxScheme>

                </cac:TaxCategory>
            </cac:TaxSubtotal>
        </cac:TaxTotal>
    </xsl:template>


</xsl:stylesheet>
