<?xml version="1.0" encoding="UTF-8"?>
<!--
******************************************************************************************************************

	Transformation from OIOUBL 3 to OIOUBL 3

	Publisher:         Erhvervsstyrelsen
	Created date:      2024-09-30
	Last changed date: 

	Description:       Transform OIOUBL 2.1 Invoice or Credit note to a valid OIOUBL 3  
	Rights:            It can be used following the Common Creative License

    Stylesheet version: 3.2.3.0e92060a

	For more information contact Erhvervsstyrelsen
==================================================================================================================

OVERVIEW:
===========
This XSLT 2.0 transformation converts OIOUBL 2.1 Invoice and CreditNote documents into OIOUBL 3.0-compliant XML 
documents. The stylesheet ensures compatibility with newer OIOUBL 3.0 standards, such as updated 
CustomizationID/ProfileID values, mandatory field defaults, and Peppol interoperability adjustments 
(e.g. calculated TaxInclusiveAmount, standardized TaxCategory codes, and cleaned company identifiers).

USAGE:
===========
Applies templates based on root node: 'Invoice' or 'CreditNote'.
Outputs transformed OIOUBL 3.0 structure with proper namespaces and schema conformance.

MAIN STRUCTURE:
===========
1. Root Template:
   - Determines document type and applies correct processing mode (Invoice/CreditNote).

2. Document Templates:
   - <Invoice> and <CreditNote>: Creates top-level structure, invokes header and line processors.

3. Header Construction (`CreateDocumentHeader`):
   - Copies or constructs mandatory UBL header fields.
   - Inserts missing values like `DueDate`, default `CustomizationID`, and calculated `TaxInclusiveAmount`.
   - Adds BuyerReference from nested Contact/ID.
   - Filters out `DocumentReference` from `OrderReference`.

4. Line Construction (`createDocumentLines`):
   - Copies and transforms line-level content, including pricing, tax, references, and classification.

5. PaymentMeans Templates:
   - Handles different combinations of `PaymentMeansCode` and `PaymentChannelCode`.
   - Maps fields like financial account info, mandates, payment IDs.

6. Party Templates:
   - Transforms party entities (Supplier, Customer, Payee, etc.).
   - Normalizes company IDs and contact details (e.g., prepends '00' to phone numbers if needed).

7. Tax, Address, and Delivery Templates:
   - Standardizes TaxCategory/ID (e.g. 'StandardRated' → 'S').
   - Ensures fallback/defaults for missing address fields using values from `AccountingCustomerParty`.

8. Item Templates:
   - Handles descriptions, classifications, and identifiers.
   - Fallback logic for `ClassifiedTaxCategory` based on associated tax details.
   - Maps embedded document references and specifications.

9. Utility Templates:
   - Converts schemeID attributes to standard ISO 6523 codes.
   - Rewrites DocumentReference/Attachment data with default filenames when necessary.
   - Ensures that missing or inconsistent values are handled with defaults or warnings as comments.


==================================================================================================================
-->


<xsl:stylesheet version="2.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:cac="urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2"
    xmlns:cbc="urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2"
    xmlns:ccts="urn:oasis:names:specification:ubl:schema:xsd:CoreComponentParameters-2"
    xmlns:cec="urn:oasis:names:specification:ubl:schema:xsd:CommonExtensionComponents-2"
    xmlns:sdt="urn:oasis:names:specification:ubl:schema:xsd:SpecializedDatatypes-2"
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    xmlns:ext="urn:oasis:names:specification:ubl:schema:xsd:CommonExtensionComponents-2"
    xmlns:udt="urn:un:unece:uncefact:data:specification:UnqualifiedDataTypesSchemaModule:2"
    xmlns:xs="http://www.w3.org/2001/XMLSchema">
    <xsl:output method="xml" encoding="UTF-8" indent="yes"/>

    <!--  ********************
    */
    */  Global values
    */
    **************************  -->
    <xsl:variable name="rootName" select="name(/*)"/>

    <xsl:variable name="root" select="/*"/>


    <!--  ********************
    */
    */  Start: Identify documenttype (Invoice or CreditNote)
    */
    **************************  -->
    <!--
    ROOT TEMPLATE:
    Selects the root node and determines if the input document is an Invoice or CreditNote.
    Applies the appropriate processing mode accordingly.
    -->
    <xsl:template match="/">

        <xsl:choose>

            <xsl:when test="$rootName = 'Invoice'">

                <xsl:apply-templates mode="Invoice"/>

            </xsl:when>

            <xsl:when test="$rootName = 'CreditNote'">

                <xsl:apply-templates mode="CreditNote"/>

            </xsl:when>

            <xsl:otherwise>

                <xsl:text>Fatal error: Unsupported documenttype! This stylesheet only supports conversion of a OIOUBL2.1 Invoice or CreditNote.</xsl:text>

            </xsl:otherwise>

        </xsl:choose>
    </xsl:template>


    <!--  ********************
    */
    */ Create the main structure for the document type and process the transformation
    */
    **************************  -->

    <!-- Invoice -->
    <!--
    INVOICE TEMPLATE:
    Builds the root <Invoice> element in the OIOUBL 3.0 namespace.
    Calls header and line transformation routines.
    -->
    <xsl:template match="*[local-name() = 'Invoice']" mode="Invoice">

        <Invoice xmlns="urn:oasis:names:specification:ubl:schema:xsd:Invoice-2">

            <!-- Transform the invoice header information -->
            <xsl:call-template name="CreateDocumentHeader"/>

            <!-- Transform the invoice line information -->
            <xsl:apply-templates select="cac:InvoiceLine"/>

        </Invoice>
    </xsl:template>

    <!-- Create InvoiceLine elemnent per line -->
    <xsl:template match="cac:InvoiceLine">

        <cac:InvoiceLine>
            <xsl:call-template name="createDocumentLines"/>
        </cac:InvoiceLine>

    </xsl:template>


    <!-- Credit Note -->
    <!--
    CREDIT NOTE TEMPLATE:
    Builds the root <CreditNote> element in the OIOUBL 3.0 namespace.
    Calls header and line transformation routines.
    -->
    <xsl:template match="*[local-name() = 'CreditNote']" mode="CreditNote">

        <CreditNote xmlns="urn:oasis:names:specification:ubl:schema:xsd:CreditNote-2">

            <!-- Transform the CreditNote header information -->
            <xsl:call-template name="CreateDocumentHeader"/>

            <!-- Transform the credit note line information -->
            <xsl:apply-templates select="cac:CreditNoteLine"/>

        </CreditNote>
    </xsl:template>

    <!-- Create CreditNoteLine elemnent per line -->
    <xsl:template match="cac:CreditNoteLine">

        <cac:CreditNoteLine>
            <xsl:call-template name="createDocumentLines"/>
        </cac:CreditNoteLine>

    </xsl:template>


    <!--  ********************
    */
    */ Create the header content for the document type
    */
    **************************  -->

    <!--
    CREATE DOCUMENT HEADER:
    Builds the document header with UBLVersionID, CustomizationID, dates, currencies, references,
    supplier/customer information, and LegalMonetaryTotal including recalculated tax totals if needed.
    -->
    <xsl:template name="CreateDocumentHeader">

        <cbc:UBLVersionID>2.1</cbc:UBLVersionID>

        <xsl:choose>
            <xsl:when test="$rootName = 'Invoice'">

                <cbc:CustomizationID>urn:fdc:oioubl.dk:trns:billing:invoice:3.0</cbc:CustomizationID>

            </xsl:when>
            <xsl:otherwise>

                <cbc:CustomizationID>urn:fdc:oioubl.dk:trns:billing:creditnote:3.0</cbc:CustomizationID>

            </xsl:otherwise>
        </xsl:choose>

        <cbc:ProfileID>urn:fdc:oioubl.dk:bis:billing_with_response:3</cbc:ProfileID>

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:CopyIndicator" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:UUID" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:IssueDate" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:IssueTime" copy-namespaces="no"/>

        <!-- Ensure DueDate on header -->
        <xsl:if test="$rootName = 'Invoice'">
            <xsl:choose>
                <xsl:when test="cac:PaymentMeans[cbc:PaymentDueDate]">

                    <xsl:comment>DueDate inserted from PaymentMeans</xsl:comment>

                    <cbc:DueDate>
                        <xsl:value-of select="cac:PaymentMeans[1]/cbc:PaymentDueDate"/>
                    </cbc:DueDate>

                </xsl:when>
                <xsl:otherwise>

                    <xsl:comment>Default cbc:DueDate inserted</xsl:comment>

                    <cbc:DueDate>1980-01-01</cbc:DueDate>

                </xsl:otherwise>
            </xsl:choose>
        </xsl:if>

        <!-- Ensure correct TypeCode -->
        <xsl:call-template name="typeCode"/>

        <xsl:copy-of select="cbc:Note" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:TaxPointDate" copy-namespaces="no"/>

        <xsl:apply-templates select="cbc:DocumentCurrencyCode"/>

        <xsl:apply-templates select="cbc:TaxCurrencyCode"/>

        <xsl:apply-templates select="cbc:PricingCurrencyCode"/>

        <xsl:copy-of select="cbc:AccountingCost" copy-namespaces="no"/>

        <xsl:comment>BuyerReference created with information from AccountingCustomerParty/Party/Contact/ID</xsl:comment>
        <cbc:BuyerReference>
            <xsl:value-of select="cac:AccountingCustomerParty/cac:Party/cac:Contact/cbc:ID"/>
        </cbc:BuyerReference>

        <xsl:copy-of select="cbc:InvoicePeriod" copy-namespaces="no"/>

        <!-- Copy all elements in OrderReference except DocumentReference -->
        <xsl:if test="cac:OrderReference">

            <cac:OrderReference>
                <xsl:copy-of select="cac:OrderReference/*[name() != 'DocumentReference']"/>
            </cac:OrderReference>

        </xsl:if>

        <xsl:apply-templates select="cac:BillingReference"/>

        <xsl:apply-templates select="cac:DespatchDocumentReference" mode="customMode"/>

        <xsl:apply-templates select="cac:ReceiptDocumentReference" mode="customMode"/>

        <xsl:apply-templates select="cac:StatementDocumentReference" mode="customMode"/>

        <xsl:apply-templates select="cac:OriginatorDocumentReference" mode="customMode"/>

        <xsl:apply-templates select="cac:ContractDocumentReference" mode="customMode"/>

        <xsl:apply-templates select="cac:AdditionalDocumentReference" mode="customMode"/>

        <xsl:copy-of select="cac:Signature" copy-namespaces="no"/>

        <xsl:apply-templates select="cac:AccountingSupplierParty"/>

        <xsl:apply-templates select="cac:AccountingCustomerParty"/>

        <xsl:apply-templates select="cac:PayeeParty"/>

        <xsl:apply-templates select="cac:BuyerCustomerParty"/>

        <xsl:apply-templates select="cac:SellerSupplierPart"/>

        <xsl:apply-templates select="cac:OriginatorParty"/>

        <xsl:apply-templates select="cac:Delivery"/>

        <xsl:apply-templates select="cac:PaymentMeans"/>

        <xsl:copy-of select="cac:PaymentTerms" copy-namespaces="no"/>

        <xsl:copy-of select="cac:PrepaidPayment" copy-namespaces="no"/>

        <xsl:apply-templates select="cac:AllowanceCharge"/>
        <!-- <xsl:copy-of select="AllowanceCharge" copy-namespaces="no" />
-->
        <xsl:copy-of select="cac:TaxExchangeRate" copy-namespaces="no"/>

        <xsl:copy-of select="cac:PaymentExchangeRate" copy-namespaces="no"/>

        <xsl:apply-templates select="cac:TaxTotal"/>

        <xsl:copy-of select="cac:PricingExchangeRate" copy-namespaces="no"/>

        <cac:LegalMonetaryTotal>

            <xsl:variable name="currency"
                select="cac:LegalMonetaryTotal/cbc:LineExtensionAmount/@currencyID"/>

            <xsl:copy-of select="cac:LegalMonetaryTotal/cbc:LineExtensionAmount"
                copy-namespaces="no"/>

            <xsl:comment>TaxExclusiveAmount is recalculated to align with Peppol</xsl:comment>
            <xsl:variable name="TaxExclusiveAmount" select="
                    format-number(
                    (if (cac:LegalMonetaryTotal/cbc:LineExtensionAmount)
                    then
                        number(cac:LegalMonetaryTotal/cbc:LineExtensionAmount)
                    else
                        0)
                    -
                    (if (cac:LegalMonetaryTotal/cac:AllowanceTotalAmount)
                    then
                        number(cac:LegalMonetaryTotal/cac:AllowanceTotalAmount)
                    else
                        0)
                    +
                    (if (cac:LegalMonetaryTotal/cbc:ChargeTotalAmount)
                    then
                        number(cac:LegalMonetaryTotal/cbc:ChargeTotalAmount)
                    else
                        0),
                    '0.00')"> </xsl:variable>

            <cbc:TaxExclusiveAmount currencyID="{$currency}">

                <xsl:value-of select="$TaxExclusiveAmount"/>

            </cbc:TaxExclusiveAmount>

            <xsl:variable name="Vat" select="
                    sum($root/cac:TaxTotal/cbc:TaxAmount[@currencyID = $root/cac:LegalMonetaryTotal/cbc:LineExtensionAmount/@currencyID])"/>

            <xsl:choose>

                <xsl:when test="not(cac:LegalMonetaryTotal/cbc:TaxInclusiveAmount)">

                    <xsl:comment>TaxInclusiveAmount is calculated as it's missing the the source document</xsl:comment>

                    <cbc:TaxInclusiveAmount currencyID="{$currency}">
                        <xsl:value-of select="
                                format-number(
                                xs:decimal($TaxExclusiveAmount)
                                + xs:decimal($Vat)
                                , '0.00')"/>
                    </cbc:TaxInclusiveAmount>

                </xsl:when>
                <xsl:otherwise>

                    <xsl:copy-of select="cac:LegalMonetaryTotal/cbc:TaxInclusiveAmount"
                        copy-namespaces="no"/>

                </xsl:otherwise>
            </xsl:choose>

            <xsl:copy-of select="cac:LegalMonetaryTotal/cbc:AllowanceTotalAmount"
                copy-namespaces="no"/>

            <xsl:copy-of select="cac:LegalMonetaryTotal/cbc:ChargeTotalAmount" copy-namespaces="no"/>

            <xsl:copy-of select="cac:LegalMonetaryTotal/cbc:PrepaidAmount" copy-namespaces="no"/>

            <xsl:copy-of select="cac:LegalMonetaryTotal/cbc:PayableRoundingAmount"
                copy-namespaces="no"/>

            <xsl:copy-of select="cac:LegalMonetaryTotal/cbc:PayableAmount" copy-namespaces="no"/>

        </cac:LegalMonetaryTotal>

    </xsl:template>


    <!--  ********************
    */
    */ Create the line content for the document type
    */
    **************************  -->

    <!--
    CREATE DOCUMENT LINES:
    Processes <InvoiceLine> or <CreditNoteLine> entries and transforms line-level data including quantities,
    taxes, item details, pricing and references.
    -->
    <xsl:template name="createDocumentLines">

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:Note" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:InvoicedQuantity" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:CreditedQuantity" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:LineExtensionAmount" copy-namespaces="no"/>

        <xsl:copy-of select="cac:TaxPointDate" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:AccountingCostCode" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:AccountingCost" copy-namespaces="no"/>

        <xsl:apply-templates select="cac:OrderLineReference"/>

        <xsl:apply-templates select="cac:DespatchLineReference"/>

        <xsl:apply-templates select="cac:ReceiptLineReference"/>

        <xsl:copy-of select="cbc:PricingReference" copy-namespaces="no"/>

        <xsl:apply-templates select="cac:DocumentReference" mode="customMode"/>

        <xsl:apply-templates select="cac:OriginatorParty"/>

        <xsl:apply-templates select="cac:Delivery"/>

        <xsl:apply-templates select="cac:AllowanceCharge"/>
        <!--    <xsl:copy-of select="cac:AllowanceCharge" copy-namespaces="no" />-->

        <xsl:apply-templates select="cac:TaxTotal"/>

        <cac:Item>
            <xsl:apply-templates select="cac:Item"/>
        </cac:Item>

        <xsl:apply-templates select="cac:Price"/>

    </xsl:template>


    <!--  ********************
    */
    */ Support templates 
    */ 
    **************************  -->

    <!-- Create Party -->

    <xsl:template
        match="cac:ManufacturerParty | cac:AccountingSupplierParty | cac:AccountingCustomerParty | cac:PayeeParty | cac:BuyerCustomerParty | cac:SellerSupplierParty | cac:OriginatorParty">

        <xsl:element name="{name()}">

            <cac:Party>

                <xsl:apply-templates select="cac:Party/cbc:EndpointID"/>

                <xsl:if test="cac:Party/cac:PartyIdentification">

                    <cac:PartyIdentification>
                        <xsl:apply-templates select="cac:Party/cac:PartyIdentification/cbc:ID"/>
                    </cac:PartyIdentification>

                </xsl:if>

                <xsl:if test="cac:Party/cac:PartyName/cbc:Name">

                    <cac:PartyName>
                        <cbc:Name>
                            <xsl:value-of select="cac:Party/cac:PartyName/cbc:Name"/>
                        </cbc:Name>
                    </cac:PartyName>

                </xsl:if>

                <cac:PostalAddress>
                    <xsl:apply-templates select="cac:Party/cac:PostalAddress"/>
                </cac:PostalAddress>

                <xsl:apply-templates select="cac:Party/cac:PartyTaxScheme"/>

                <xsl:apply-templates select="cac:Party/cac:PartyLegalEntity"/>

                <xsl:apply-templates select="cac:Party/cac:Contact"/>

            </cac:Party>

        </xsl:element>

    </xsl:template>

    <!--  ********************
    */
    */ Template: Handling Payment means
    */ 
    **************************  -->

    <!-- Find PaymentMeans -->
    <!--
    PAYMENT MEANS TEMPLATE:
    Determines and processes payment information based on PaymentMeansCode and PaymentChannelCode.
    Calls specialized named templates for specific cases (IBAN, bank, direct debit, etc.).
    -->
    <xsl:template match="cac:PaymentMeans">

        <xsl:choose>
            <xsl:when test="(cbc:PaymentMeansCode = '31') and cbc:PaymentChannelCode = 'IBAN'">
                <cac:PaymentMeans>
                    <xsl:call-template name="paymentMeans_31"/>
                </cac:PaymentMeans>
            </xsl:when>

            <xsl:when test="cbc:PaymentMeansCode = '31' and cbc:PaymentChannelCode = 'ZZZ'">
                <cac:PaymentMeans>
                    <xsl:call-template name="paymentMeans_31_ZZZ"/>
                </cac:PaymentMeans>
            </xsl:when>

            <xsl:when test="(cbc:PaymentMeansCode = '42') and cbc:PaymentChannelCode = 'DK:BANK'">
                <cac:PaymentMeans>
                    <xsl:call-template name="paymentMeans_42"/>
                </cac:PaymentMeans>
            </xsl:when>

            <xsl:when test="cbc:PaymentMeansCode = '49'">
                <cac:PaymentMeans>
                    <xsl:call-template name="paymentMeans_49"/>
                </cac:PaymentMeans>
            </xsl:when>

            <xsl:when test="cbc:PaymentMeansCode = '50'">
                <cac:PaymentMeans>
                    <xsl:call-template name="paymentMeans_50"/>
                </cac:PaymentMeans>
            </xsl:when>

            <xsl:when test="cbc:PaymentMeansCode = '93'">
                <cac:PaymentMeans>
                    <xsl:call-template name="paymentMeans_93"/>
                </cac:PaymentMeans>
            </xsl:when>

            <xsl:when test="cbc:PaymentMeansCode = '58'">
                <cac:PaymentMeans>
                    <xsl:call-template name="paymentMeans_58"/>
                </cac:PaymentMeans>
            </xsl:when>

            <xsl:when test="cbc:PaymentMeansCode = '59'">
                <cac:PaymentMeans>
                    <xsl:call-template name="paymentMeans_59"/>
                </cac:PaymentMeans>
            </xsl:when>

            <xsl:when test="cbc:PaymentMeansCode = '97'">
                <cac:PaymentMeans>
                    <xsl:call-template name="paymentMeans_97"/>
                </cac:PaymentMeans>
            </xsl:when>

            <xsl:otherwise>
                <cac:PaymentMeans>
                    <xsl:value-of select="'PaymentMeans error'"/>
                </cac:PaymentMeans>
            </xsl:otherwise>
        </xsl:choose>

    </xsl:template>

    <!--  ********************
    */
    */ Template to create the specific PaymentMeans
    */
    **************************  -->
    <!-- PaymentMeansCode = 31 -->

    <xsl:template name="paymentMeans_31">

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <cbc:PaymentMeansCode>
            <xsl:value-of select="cbc:PaymentMeansCode"/>
        </cbc:PaymentMeansCode>

        <cbc:PaymentChannelCode>
            <xsl:value-of select="cbc:PaymentChannelCode"/>
        </cbc:PaymentChannelCode>

        <cac:PayeeFinancialAccount>

            <xsl:copy-of select="cac:PayeeFinancialAccount/cbc:ID" copy-namespaces="no"/>

            <xsl:copy-of select="cac:PayeeFinancialAccount/PaymentNote" copy-namespaces="no"/>

            <xsl:copy-of select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch"
                copy-namespaces="no"/>

        </cac:PayeeFinancialAccount>
    </xsl:template>

    <!-- PaymentMeansCode = 31 and cbc:PaymentChannelCode = ZZZ -->
    <xsl:template name="paymentMeans_31_ZZZ">

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <cbc:PaymentMeansCode>
            <xsl:value-of select="cbc:PaymentMeansCode"/>
        </cbc:PaymentMeansCode>

        <cbc:PaymentChannelCode>
            <xsl:value-of select="cbc:PaymentChannelCode"/>
        </cbc:PaymentChannelCode>

        <cac:PayeeFinancialAccount>

            <xsl:copy-of select="cac:PayeeFinancialAccount/cbc:ID" copy-namespaces="no"/>

            <xsl:copy-of select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch"
                copy-namespaces="no"/>

        </cac:PayeeFinancialAccount>
    </xsl:template>

    <!-- PaymentMeansCode = 42 -->

    <xsl:template name="paymentMeans_42">

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <cbc:PaymentMeansCode>
            <xsl:value-of select="cbc:PaymentMeansCode"/>
        </cbc:PaymentMeansCode>

        <cbc:PaymentChannelCode>
            <xsl:value-of select="cbc:PaymentChannelCode"/>
        </cbc:PaymentChannelCode>

        <cac:PayeeFinancialAccount>

            <xsl:copy-of select="cac:PayeeFinancialAccount/cbc:ID" copy-namespaces="no"/>

            <xsl:copy-of select="cac:PayeeFinancialAccount/cbc:PaymentNote" copy-namespaces="no"/>

            <xsl:copy-of select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch"
                copy-namespaces="no"/>

        </cac:PayeeFinancialAccount>
    </xsl:template>

    <!-- PaymentMeansCode = 49 (Direct debit)-->
    <xsl:template name="paymentMeans_49">

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>
        <xsl:copy-of select="cbc:PaymentMeansCode" copy-namespaces="no"/>
        <xsl:copy-of select="cbc:PaymentChannelCode" copy-namespaces="no"/>
        <xsl:copy-of select="cbc:PaymentID" copy-namespaces="no"/>
        <xsl:copy-of select="cac:CardAccount" copy-namespaces="no"/>
        <xsl:copy-of select="cac:CreditAccount" copy-namespaces="no"/>

        <xsl:if test="cac:PayerFinancialAccount">
            
            <cac:PaymentMandate>
                
                <cbc:ID>
                    <xsl:copy-of select="cbc:InstructionID" copy-namespaces="no"/>
                </cbc:ID>


                <xsl:copy-of select="cac:PayerFinancialAccount" copy-namespaces="no"/>

            </cac:PaymentMandate>
        </xsl:if>


    </xsl:template>





    <!-- PaymentMeansCode 93 and PaymentID = 71, 73, 75 -->
    <xsl:template name="paymentMeans_93">

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <cbc:PaymentMeansCode>
            <xsl:value-of select="cbc:PaymentMeansCode"/>
        </cbc:PaymentMeansCode>

        <xsl:copy-of select="cbc:InstructionID" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:InstructionNote" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:PaymentID" copy-namespaces="no"/>

        <cac:CreditAccount>
            <xsl:copy-of select="cac:CreditAccount/cbc:AccountID" copy-namespaces="no"/>
        </cac:CreditAccount>

    </xsl:template>

    <!-- PaymentMeansCode 50 and PaymentID = 01, 04, 15 -->
    <xsl:template name="paymentMeans_50">

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <cbc:PaymentMeansCode>
            <xsl:value-of select="cbc:PaymentMeansCode"/>
        </cbc:PaymentMeansCode>

        <xsl:copy-of select="cbc:InstructionID" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:InstructionNote" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:PaymentID" copy-namespaces="no"/>

        <cac:PayeeFinancialAccount>
            <xsl:copy-of select="cac:PayeeFinancialAccount/cbc:ID" copy-namespaces="no"/>
        </cac:PayeeFinancialAccount>

    </xsl:template>

    <!-- PaymentMeansCode = 58 -->
    <xsl:template name="paymentMeans_58">

        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <cbc:PaymentMeansCode>
            <xsl:value-of select="cbc:PaymentMeansCode"/>
        </cbc:PaymentMeansCode>

        <xsl:copy-of select="cbc:PaymentChannelCode" copy-namespaces="no"/>


        <xsl:if test="cac:PayeeFinancialAccount">

            <cac:PayeeFinancialAccount>

                <xsl:copy-of select="cac:PayeeFinancialAccount/cbc:ID" copy-namespaces="no"/>

                <xsl:copy-of select="cac:PayeeFinancialAccount/cbc:Name" copy-namespaces="no"/>

                <xsl:copy-of select="cac:PayeeFinancialAccount/cbc:PaymentNote" copy-namespaces="no"/>


                <xsl:if test="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch">

                    <cac:FinancialInstitutionBranch>

                        <xsl:copy-of
                            select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cbc:ID"
                            copy-namespaces="no"/>

                        <xsl:copy-of
                            select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cbc:Name"
                            copy-namespaces="no"/>

                        <xsl:if
                            test="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cac:Address">
                            <cac:Address>
                                <xsl:apply-templates
                                    select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cac:Address"
                                />
                            </cac:Address>
                        </xsl:if>


                    </cac:FinancialInstitutionBranch>

                </xsl:if>

            </cac:PayeeFinancialAccount>

        </xsl:if>



    </xsl:template>


    <!-- PaymentMeansCode = 59 -->
    <xsl:template name="paymentMeans_59">
        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <cbc:PaymentMeansCode>
            <xsl:value-of select="cbc:PaymentMeansCode"/>
        </cbc:PaymentMeansCode>

        <cbc:PaymentChannelCode>
            <xsl:value-of select="cbc:PaymentChannelCode"/>
        </cbc:PaymentChannelCode>

        <cac:PayeeFinancialAccount>

            <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

            <xsl:copy-of select="cac:PayeeFinancialAccount/cbc:PaymentNote" copy-namespaces="no"/>

            <xsl:copy-of select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch"
                copy-namespaces="no"/>
        </cac:PayeeFinancialAccount>

        <xsl:copy-of select="cac:PaymentMandate" copy-namespaces="no"/>

    </xsl:template>

    <xsl:template name="paymentMeans_97">
        <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

        <cbc:PaymentMeansCode>
            <xsl:value-of select="cbc:PaymentMeansCode"/>
        </cbc:PaymentMeansCode>

        <xsl:if test="cbc:PaymentChannelCode">
            <cbc:PaymentChannelCode>
                <xsl:value-of select="cbc:PaymentChannelCode"/>
            </cbc:PaymentChannelCode>
        </xsl:if>

    </xsl:template>

    <!--  ********************
    */
    */ Templates for creation segments
    */ 
    **************************  -->
    <!-- Delivery segment -->
    <xsl:template match="cac:Delivery">

        <cac:Delivery>

            <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

            <xsl:copy-of select="cbc:Quantity" copy-namespaces="no"/>

            <xsl:copy-of select="cbc:MinimumQuantity" copy-namespaces="no"/>

            <xsl:copy-of select="cbc:MaximumQuantity" copy-namespaces="no"/>

            <xsl:copy-of select="cbc:ActualDeliveryDate" copy-namespaces="no"/>

            <xsl:copy-of select="cbc:ActualDeliveryTime" copy-namespaces="no"/>

            <xsl:copy-of select="cbc:LatestDeliveryDate" copy-namespaces="no"/>

            <xsl:copy-of select="cbc:LatestDeliveryTime" copy-namespaces="no"/>

            <xsl:if test="string(cac:DeliveryLocation)">

                <cac:DeliveryLocation>
                    <cac:Address>
                        <xsl:apply-templates select="cac:DeliveryLocation/cac:Address"/>
                    </cac:Address>
                </cac:DeliveryLocation>
            </xsl:if>

            <xsl:if test="string(cac:RequestedDeliveryPeriod)">

                <cac:RequestedDeliveryPeriod>
                    <xsl:apply-templates select="cac:Period"/>
                </cac:RequestedDeliveryPeriod>

            </xsl:if>

            <xsl:if test="string(cac:PromisedDeliveryPeriod)">

                <cac:PromisedDeliveryPeriod>
                    <xsl:apply-templates select="cac:Period"/>
                </cac:PromisedDeliveryPeriod>

            </xsl:if>

            <xsl:if test="string(cac:EstimatedDeliveryPeriod)">

                <cac:EstimatedDeliveryPeriod>
                    <xsl:apply-templates select="cac:Period"/>

                </cac:EstimatedDeliveryPeriod>

            </xsl:if>

            <xsl:if test="string(cac:DeliveryParty)">

                <cac:DeliveryParty>

                    <xsl:copy-of select="cac:DeliveryParty/cbc:EndpointID" copy-namespaces="no"/>

                    <xsl:if test="cac:DeliveryParty/cac:PartyIdentification">

                        <cac:PartyIdentification>
                            <xsl:apply-templates
                                select="cac:DeliveryParty/cac:PartyIdentification/cbc:ID"/>
                        </cac:PartyIdentification>

                    </xsl:if>

                    <xsl:copy-of select="cac:DeliveryParty/cac:PartyName" copy-namespaces="no"/>

                    <xsl:apply-templates select="cac:DeliveryParty/cac:PostalAddress"/>

                    <xsl:copy-of select="cac:DeliveryParty/cac:PartyLegalEntity"
                        copy-namespaces="no"/>

                    <xsl:copy-of select="cac:DeliveryParty/cac:Contact" copy-namespaces="no"/>


                </cac:DeliveryParty>
            </xsl:if>
        </cac:Delivery>
    </xsl:template>

    <!-- Period segment -->
    <xsl:template match="cac:Period">

        <xsl:copy-of select="cbc:StartDate" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:StartTime" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:EndDate" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:EndTime" copy-namespaces="no"/>

        <xsl:copy-of select="cbc:Description" copy-namespaces="no"/>
    </xsl:template>

    <!-- Address segment -->
    <xsl:template
        match="cac:PostalAddress | cac:DeliveryLocation/cac:Address | cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cac:Address">

        <xsl:choose>
            <xsl:when test="cbc:StreetName">

                <xsl:copy-of select="cbc:StreetName" copy-namespaces="no"/>

            </xsl:when>
            <xsl:otherwise>

                <xsl:comment>Default StreetName inserted from cac:AccountingCustomerParty</xsl:comment>
                <cbc:StreetName>
                    <xsl:value-of
                        select="//cac:AccountingCustomerParty/cac:Party/cac:PostalAddress/cbc:StreetName"
                    />
                </cbc:StreetName>

            </xsl:otherwise>
        </xsl:choose>
        <xsl:choose>

            <xsl:when test="cbc:BuildingNumber">
                <xsl:copy-of select="cbc:BuildingNumber" copy-namespaces="no"/>
            </xsl:when>

            <xsl:otherwise>

                <xsl:comment>Default BuildingNumber inserted</xsl:comment>
                <cbc:BuildingNumber>Default</cbc:BuildingNumber>

            </xsl:otherwise>
        </xsl:choose>
        <xsl:choose>

            <xsl:when test="cbc:CityName">
                <xsl:copy-of select="cbc:CityName" copy-namespaces="no"/>
            </xsl:when>

            <xsl:otherwise>
                <xsl:comment>Default CityName inserted from cac:AccountingCustomerParty</xsl:comment>
                <cbc:CityName>
                    <xsl:value-of
                        select="//cac:AccountingCustomerParty/cac:Party/cac:PostalAddress/cbc:CityName"
                    />
                </cbc:CityName>

            </xsl:otherwise>
        </xsl:choose>
        <xsl:choose>

            <xsl:when test="cbc:PostalZone">
                <xsl:copy-of select="cbc:PostalZone" copy-namespaces="no"/>
            </xsl:when>

            <xsl:otherwise>

                <xsl:comment>Default PostalZone inserted from cac:AccountingCustomerParty</xsl:comment>
                <cbc:PostalZone>
                    <xsl:value-of
                        select="//cac:AccountingCustomerParty/cac:Party/cac:PostalAddress/cbc:PostalZone"
                    />
                </cbc:PostalZone>

            </xsl:otherwise>
        </xsl:choose>


        <xsl:choose>

            <xsl:when test="cac:Country">
                <xsl:copy-of select="cac:Country" copy-namespaces="no"/>
            </xsl:when>

            <xsl:otherwise>

                <xsl:comment>Default Country inserted from cac:AccountingCustomerParty</xsl:comment>
                <cac:Country>

                    <cbc:IdentificationCode>
                        <xsl:value-of
                            select="//cac:AccountingCustomerParty/cac:Party/cac:PostalAddress/cac:Country/cbc:IdentificationCode"
                        />
                    </cbc:IdentificationCode>
                    <cbc:Name>Default</cbc:Name>

                </cac:Country>

            </xsl:otherwise>
        </xsl:choose>

    </xsl:template>

    <!-- Tax segment -->
    <!--
    TAX TOTAL TEMPLATE:
    Transforms tax summary data and maps TaxCategory values to Peppol-compliant codes.
    Handles fallback logic for tax percentages and exemption codes.
    -->
    <xsl:template match="cac:TaxTotal">

        <cac:TaxTotal>

            <cbc:TaxAmount currencyID="{cbc:TaxAmount/@currencyID}">
                <xsl:value-of select="cbc:TaxAmount"/>

            </cbc:TaxAmount>
            <xsl:for-each select="cac:TaxSubtotal">

                <cac:TaxSubtotal>

                    <cbc:TaxableAmount currencyID="{cbc:TaxableAmount/@currencyID}">
                        <xsl:value-of select="cbc:TaxableAmount"/>
                    </cbc:TaxableAmount>

                    <cbc:TaxAmount currencyID="{cbc:TaxAmount/@currencyID}">
                        <xsl:value-of select="cbc:TaxAmount"/>
                    </cbc:TaxAmount>

                    <xsl:comment>The value of TaxCategory/ID is changed to support Peppol notation</xsl:comment>
                    <cac:TaxCategory>

                        <cbc:ID>
                            <xsl:value-of select="
                                    if (cac:TaxCategory/cbc:ID = 'StandardRated') then
                                        'S'
                                    else
                                        if (cac:TaxCategory/cbc:ID = 'ZeroRated') then
                                            'Z'
                                        else
                                            if (cac:TaxCategory/cbc:ID = 'ReverseCharge') then
                                                'AE'
                                            else
                                                cac:TaxCategory/cbc:ID"
                            />
                        </cbc:ID>

                        <xsl:choose>
                            <xsl:when test="cac:TaxCategory/cbc:Percent">

                                <cbc:Percent>
                                    <xsl:value-of select="cac:TaxCategory/cbc:Percent"/>
                                </cbc:Percent>

                            </xsl:when>


                            <xsl:when test="cac:TaxCategory/cbc:ID = 'ZeroRated'">
                                <xsl:comment>Percent inserted based on TaxCategory/ID = 'ZeroRated'</xsl:comment>
                                <cbc:Percent>0</cbc:Percent>
                            </xsl:when>

                            <xsl:otherwise>

                                <xsl:comment>Procent missing in source</xsl:comment>
                                <cbc:Percent/>

                            </xsl:otherwise>
                        </xsl:choose>

                        <xsl:if test="cac:TaxCategory/cbc:ID = 'ReverseCharge'">
                            <xsl:comment>Default TaxExemptionReasonCode inserted when Reverse charge</xsl:comment>
                            <cbc:TaxExemptionReasonCode>VATEX-EU-AE</cbc:TaxExemptionReasonCode>
                        </xsl:if>


                        <cac:TaxScheme>
                            <cbc:ID>VAT</cbc:ID>
                        </cac:TaxScheme>

                    </cac:TaxCategory>

                </cac:TaxSubtotal>

            </xsl:for-each>
        </cac:TaxTotal>
    </xsl:template>

    <!-- Item segment -->
    <!--
    ITEM TEMPLATE:
    Processes item-level data including names, identifiers, tax category classification, item properties,
    and manufacturer details. Adds fallback if no ClassifiedTaxCategory is provided.
    -->
    <xsl:template match="cac:Item">

        <xsl:if test="cbc:Description">

            <cbc:Description>
                <xsl:value-of select="cbc:Description"/>
            </cbc:Description>

        </xsl:if>

        <xsl:if test="cbc:CatalogueIndicator">

            <cbc:CatalogueIndicator>
                <xsl:value-of select="cbc:CatalogueIndicator"/>
            </cbc:CatalogueIndicator>

        </xsl:if>

        <xsl:if test="cbc:Name">

            <cbc:Name>
                <xsl:value-of select="cbc:Name"/>
            </cbc:Name>

        </xsl:if>


        <xsl:apply-templates select="cac:BuyersItemIdentification"/>

        <xsl:apply-templates select="cac:SellersItemIdentification"/>

        <xsl:apply-templates select="cac:ManufacturersItemIdentification"/>

        <xsl:apply-templates select="cac:StandardItemIdentification"/>

        <xsl:apply-templates select="cac:CatalogueItemIdentification"/>

        <xsl:apply-templates select="cac:AdditionalItemIdentification"/>

        <xsl:if test="cac:CatalogueDocumentReference">

            <cac:CatalogueDocumentReference>
                <xsl:apply-templates select="cac:CatalogueDocumentReference"/>
            </cac:CatalogueDocumentReference>

        </xsl:if>

        <xsl:if test="cac:ItemSpecificationDocumentReference">

            <cac:ItemSpecificationDocumentReference>
                <xsl:apply-templates select="cac:ItemSpecificationDocumentReference"/>
            </cac:ItemSpecificationDocumentReference>

        </xsl:if>

        <xsl:for-each select="cac:CommodityClassification">
            <xsl:copy-of select="." copy-namespaces="no"/>

        </xsl:for-each>

        <xsl:choose>
            <xsl:when test="cac:ClassifiedTaxCategory">

                <cac:ClassifiedTaxCategory>

                    <cbc:ID>
                        <xsl:value-of select="
                                if (cac:TaxCategory/cbc:ID = 'StandardRated') then
                                    'S'
                                else
                                    if (cac:TaxCategory/cbc:ID = 'ZeroRated') then
                                        'Z'
                                    else
                                        if (cac:TaxCategory/cbc:ID = 'ReverseCharge') then
                                            'AE'
                                        else
                                            cac:TaxCategory/cbc:ID"/>
                    </cbc:ID>

                    <xsl:copy-of select="cac:TaxCategory/cbc:Percent" copy-namespaces="no"/>


                    <xsl:if test="cac:TaxCategory/cbc:ID = 'ReverseCharge'">
                        <xsl:comment>Default TaxExemptionReasonCode inserted when Reverse charge</xsl:comment>
                        <cbc:TaxExemptionReasonCode>VATEX-EU-AE</cbc:TaxExemptionReasonCode>
                    </xsl:if>


                    <xsl:copy-of select="cac:ClassifiedTaxCategory/cac:ID" copy-namespaces="no"/>

                    <xsl:copy-of select="cac:ClassifiedTaxCategory/cac:Name" copy-namespaces="no"/>

                    <xsl:copy-of select="cac:ClassifiedTaxCategory/cac:Percent" copy-namespaces="no"/>

                    <xsl:copy-of select="cac:ClassifiedTaxCategory/cbc:TaxExemptionReasonCode"
                        copy-namespaces="no"/>

                    <xsl:copy-of select="cac:ClassifiedTaxCategory/cbc:TaxExemptionReason"
                        copy-namespaces="no"/>

                    <cac:TaxScheme>
                        <cbc:ID>VAT</cbc:ID>
                    </cac:TaxScheme>

                </cac:ClassifiedTaxCategory>
            </xsl:when>
            <xsl:otherwise>
                <xsl:comment>Default ClassifiedTaxCategory created with information from TaxCategory where possible</xsl:comment>

                <xsl:variable name="taxPercent" select="
                        if (not(../cac:TaxTotal[1]/cac:TaxSubtotal/cac:TaxCategory/cbc:Percent)) then
                            '0'
                        else
                            ../cac:TaxTotal[1]/cac:TaxSubtotal/cac:TaxCategory/cbc:Percent"/>

                <xsl:variable name="taxId" select="
                        if (../cac:TaxTotal[1]/cac:TaxSubtotal/cac:TaxCategory/cbc:ID = 'ZeroRated') then
                            'Z'
                        else
                            if (../cac:TaxTotal[1]/cac:TaxSubtotal/cac:TaxCategory/cbc:ID = 'StandardRated') then
                                'S'
                            else
                                if (../cac:TaxTotal[1]/cac:TaxSubtotal/cac:TaxCategory/cbc:ID = 'ReverseCharge') then
                                    'AE'
                                else
                                    ../cac:TaxTotal[1]/cac:TaxSubtotal/cac:TaxCategory/cbc:ID"/>

                <xsl:if test="not($taxId)">
                    <xsl:comment>No valid value for 'ClassifiedTaxCategory/ID' could be found</xsl:comment>
                </xsl:if>

                <xsl:if test="not($taxPercent)">
                    <xsl:comment>No valid value for 'ClassifiedTaxCategory/Percent' found</xsl:comment>
                </xsl:if>

                <cac:ClassifiedTaxCategory>

                    <cbc:ID>
                        <xsl:value-of select="$taxId"/>
                    </cbc:ID>

                    <cbc:Percent>
                        <xsl:value-of select="
                                if (string($taxPercent) = '' and $taxId = 'Z') then
                                    '0'
                                else
                                    $taxPercent"/>
                    </cbc:Percent>

                    <xsl:if test="cac:TaxCategory/cbc:ID = 'ReverseCharge'">
                        <xsl:comment>Default TaxExemptionReasonCode inserted when Reverse charge</xsl:comment>
                        <cbc:TaxExemptionReasonCode>VATEX-EU-AE</cbc:TaxExemptionReasonCode>
                    </xsl:if>

                    <cac:TaxScheme>
                        <cbc:ID>VAT</cbc:ID>
                    </cac:TaxScheme>

                </cac:ClassifiedTaxCategory>
            </xsl:otherwise>
        </xsl:choose>

        <xsl:for-each select="cac:AdditionalItemProperty">
            <xsl:copy-of select="." copy-namespaces="no"/>
        </xsl:for-each>

        <xsl:for-each select="cac:ManufacturerParty">

            <cac:ManufacturerParty>
                <xsl:apply-templates select="."/>
            </cac:ManufacturerParty>

        </xsl:for-each>

        <xsl:for-each select="cac:ItemInstance">

            <cac:ItemInstance>

                <xsl:if test="cbc:ProductTraceID">
                    <cac:ProductTraceID>
                        <xsl:value-of select="cbc:ProductTraceID"/>
                    </cac:ProductTraceID>

                </xsl:if>

                <xsl:if test="cbc:ManufactureDate">

                    <cac:ManufactureDate>
                        <xsl:value-of select="cbc:ManufactureDate"/>
                    </cac:ManufactureDate>

                </xsl:if>
                <xsl:if test="cbc:ManufactureTime">

                    <cbc:ManufactureTime>
                        <xsl:value-of select="cbc:ManufactureTime"/>
                    </cbc:ManufactureTime>

                </xsl:if>

                <xsl:if test="cbc:RegistrationID">

                    <cac:RegistrationID>
                        <xsl:value-of select="cbc:RegistrationID"/>
                    </cac:RegistrationID>

                </xsl:if>

                <xsl:if test="cbc:SerialID">

                    <cbc:SerialID>
                        <xsl:value-of select="cbc:SerialID"/>
                    </cbc:SerialID>

                </xsl:if>

            </cac:ItemInstance>

        </xsl:for-each>

        <xsl:apply-templates select="cac:Certificate"/>

    </xsl:template>

    <!-- Template for Certificate -->
    <xsl:template match="cac:Certificate">

        <cac:Certificate>

            <xsl:if test="cbc:ID">

                <cbc:ID>
                    <xsl:value-of select="cbc:ID"/>
                </cbc:ID>

            </xsl:if>

            <xsl:if test="cbc:CertificateTypeCode">

                <xsl:comment>Equals ID to ensure valid CertificateTypeCode</xsl:comment>
                <cbc:CertificateTypeCode>
                    <xsl:value-of select="cbc:ID"/>
                </cbc:CertificateTypeCode>

            </xsl:if>

            <xsl:if test="cbc:CertificateType">

                <cbc:CertificateType>
                    <xsl:value-of select="cbc:CertificateType"/>
                </cbc:CertificateType>

            </xsl:if>

            <xsl:if test="cbc:Remarks">

                <cbc:Remarks>
                    <xsl:value-of select="cbc:Remarks"/>

                </cbc:Remarks>
            </xsl:if>

            <xsl:if test="cac:IssuerParty">

                <cac:IssuerParty>

                    <cac:PartyName>

                        <cbc:Name>
                            <xsl:value-of select="cac:IssuerParty/cac:PartyName/cbc:Name"/>
                        </cbc:Name>

                    </cac:PartyName>

                </cac:IssuerParty>

            </xsl:if>
        </cac:Certificate>

    </xsl:template>

    <!-- Template for PartyLegalEntity -->
    <xsl:template match="cac:PartyLegalEntity">

        <cac:PartyLegalEntity>

            <xsl:if test="cbc:RegistrationName">

                <cbc:RegistrationName>
                    <xsl:value-of select="cbc:RegistrationName"/>
                </cbc:RegistrationName>

            </xsl:if>

            <xsl:comment>If DK:SE (0184) the value is stripped of any prefix eg.: DK</xsl:comment>
            <cbc:CompanyID>
                <xsl:apply-templates select="cbc:CompanyID"/>
            </cbc:CompanyID>

        </cac:PartyLegalEntity>
    </xsl:template>

    <!-- Template for PartyTaxScheme -->
    <xsl:template match="cac:PartyTaxScheme">

        <cac:PartyTaxScheme>

            <cbc:CompanyID>
                <xsl:attribute name="schemeID">
                    <xsl:choose>
                        <xsl:when test="cbc:CompanyID/@schemeID = 'DK:P'">0096</xsl:when>
                        <xsl:when test="cbc:CompanyID/@schemeID = 'DK:CVR'">0184</xsl:when>
                        <xsl:when test="cbc:CompanyID/@schemeID = 'DK:SE'">0198</xsl:when>
                        <xsl:otherwise>ZZZ</xsl:otherwise>
                    </xsl:choose>
                </xsl:attribute>

                <xsl:value-of select="cbc:CompanyID"/>

            </cbc:CompanyID>

            <cac:TaxScheme>
                <cbc:ID>VAT</cbc:ID>

                <cbc:Name>
                    <xsl:value-of select="cac:TaxScheme/cbc:Name"/>
                </cbc:Name>

            </cac:TaxScheme>

        </cac:PartyTaxScheme>
    </xsl:template>


    <!-- Template for DocumentReference -->
    <xsl:template
        match="cac:DocumentReference | cac:OriginatorParty | cac:DespatchDocumentReference | cac:ReceiptDocumentReference | cac:StatementDocumentReference | cac:OriginatorDocumentReference | cac:ContractDocumentReference | cac:AdditionalDocumentReference"
        mode="customMode">

        <xsl:element name="{name()}">
            <xsl:apply-templates
                select="cbc:ID | cbc:CopyIndicator | cbc:UUID | cbc:IssueDate | cbc:DocumentTypeCode | cac:Attachment"
                mode="customMode"/>

        </xsl:element>
    </xsl:template>

    <xsl:template match="cbc:ID" mode="customMode">

        <cbc:ID>
            <xsl:value-of select="."/>
        </cbc:ID>

    </xsl:template>

    <xsl:template match="cbc:CopyIndicator" mode="customMode">

        <cbc:CopyIndicator>
            <xsl:value-of select="."/>
        </cbc:CopyIndicator>

    </xsl:template>

    <xsl:template match="cbc:UUID" mode="customMode">

        <cbc:UUID>
            <xsl:value-of select="."/>
        </cbc:UUID>

    </xsl:template>

    <xsl:template match="cbc:IssueDate" mode="customMode">

        <cbc:IssueDate>
            <xsl:value-of select="."/>
        </cbc:IssueDate>

    </xsl:template>

    <xsl:template match="cbc:DocumentTypeCode" mode="customMode">


        <xsl:choose>

            <xsl:when test=". = '130'">

                <cbc:DocumentTypeCode>
                    <xsl:copy-of select="."/>
                </cbc:DocumentTypeCode>

            </xsl:when>
            <xsl:when test=". = '380'">

                <xsl:comment>DocumentTypeCode converted to from 380 to 130 to align with OIOUBL3</xsl:comment>
                <cbc:DocumentTypeCode>130</cbc:DocumentTypeCode>

            </xsl:when>
            <xsl:otherwise/>
        </xsl:choose>
    </xsl:template>

    <xsl:template match="cac:Attachment" mode="customMode">

        <cac:Attachment>
            <xsl:apply-templates select="cbc:EmbeddedDocumentBinaryObject | cac:ExternalReference"/>
        </cac:Attachment>

    </xsl:template>

    <!--  DocumentReference end -->


    <!-- Template for EmbeddedDocumentBinaryObject -->
    <xsl:template match="cbc:EmbeddedDocumentBinaryObject">

        <cbc:EmbeddedDocumentBinaryObject>

            <xsl:attribute name="mimeCode">
                <xsl:value-of select="@mimeCode"/>
            </xsl:attribute>

            <xsl:attribute name="filename">

                <xsl:choose>

                    <xsl:when test="@filename">
                        <xsl:value-of select="@filename"/>
                    </xsl:when>
                    <xsl:otherwise>

                        <xsl:variable name="suffix" select="substring-after(@mimeCode, '/')"/>

                        <xsl:choose>

                            <xsl:when test="$suffix = 'csv' or $suffix = 'pdf' or $suffix = 'png'">
                                <xsl:value-of select="concat('default.', $suffix)"/>
                            </xsl:when>

                            <xsl:when test="$suffix = 'jpeg'">
                                <xsl:value-of select="'default.jpg'"/>
                            </xsl:when>

                            <xsl:otherwise>Unknown.format</xsl:otherwise>

                        </xsl:choose>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:attribute>

            <xsl:value-of select="."/>

        </cbc:EmbeddedDocumentBinaryObject>

    </xsl:template>


    <!-- Template for ExternalReference -->
    <xsl:template match="cac:ExternalReference">

        <cac:ExternalReference>
            <xsl:apply-templates
                select="cbc:URI | cbc:DocumentHash | cbc:ExpiryDate | cbc:ExpiryTime"/>
        </cac:ExternalReference>

    </xsl:template>

    <xsl:template match="cbc:URI">

        <cbc:URI>
            <xsl:value-of select="."/>
        </cbc:URI>
    </xsl:template>

    <xsl:template match="cbc:DocumentHash">

        <cbc:DocumentHash>
            <xsl:value-of select="."/>
        </cbc:DocumentHash>

    </xsl:template>

    <xsl:template match="cbc:ExpiryDate">

        <cbc:ExpiryDate>
            <xsl:value-of select="."/>
        </cbc:ExpiryDate>

    </xsl:template>

    <xsl:template match="cbc:ExpiryTime">

        <cbc:ExpiryTime>
            <xsl:value-of select="."/>
        </cbc:ExpiryTime>

    </xsl:template>

    <!-- ExternalReference end -->


    <!-- Template for lineReference -->
    <xsl:template name="lineReference">

        <xsl:param name="path"/>

        <xsl:if test="string($path/cbc:LineID)">

            <cbc:LineID>
                <xsl:value-of select="$path/cbc:LineID"/>
            </cbc:LineID>

        </xsl:if>

        <xsl:if test="string($path/cbc:UUID)">

            <cbc:UUID>
                <xsl:value-of select="$path/cbc:UUID"/>
            </cbc:UUID>

        </xsl:if>

        <xsl:if test="string($path/cbc:LineStatusCode)">

            <cbc:LineStatusCode>
                <xsl:value-of select="$path/cbc:LineStatusCode"/>
            </cbc:LineStatusCode>

        </xsl:if>

        <cac:DocumentReference>
            <xsl:apply-templates select="cac:DocumentReference" mode="customMode"/>
        </cac:DocumentReference>

    </xsl:template>



    <!-- ItemIdentification segment -->
    <xsl:template match="cac:ItemIdentification">

        <xsl:if test="string(cbc:ID)">

            <cbc:ID>
                <xsl:value-of select="cbc:ID"/>
            </cbc:ID>

        </xsl:if>

        <xsl:if test="string(cbc:ExtendedID)">
            <cbc:ExtendedID>
                <xsl:value-of select="cbc:ExtendedID"/>
            </cbc:ExtendedID>

        </xsl:if>

    </xsl:template>

    <!-- ItemProperty segment -->
    <xsl:template match="cac:ItemProperty">

        <xsl:if test="cbc:Name">

            <cbc:Name>
                <xsl:value-of select="cbc:Name"/>
            </cbc:Name>

        </xsl:if>

        <xsl:if test="cbc:Value">

            <cbc:Value>
                <xsl:value-of select="cbc:Value"/>
            </cbc:Value>

        </xsl:if>

        <xsl:if test="cac:UsabilityPeriod">

            <cac:UsabilityPeriod>
                <xsl:value-of select="cac:UsabilityPeriod"/>
            </cac:UsabilityPeriod>
        </xsl:if>

        <xsl:for-each select="cac:ItemPropertyGroup">

            <cac:ItemPropertyGroup>

                <cbc:ID>
                    <xsl:value-of select="cac:ItemPropertyGroup/cbc:ID"/>
                </cbc:ID>

                <cbc:Name>
                    <xsl:value-of select="cac:ItemPropertyGroup/cbc:Name"/>
                </cbc:Name>

            </cac:ItemPropertyGroup>

        </xsl:for-each>
    </xsl:template>

    <!-- ItemIdentification segment -->
    <xsl:template match="cac:CatalogueDocumentReference | cac:ItemSpecificationDocumentReference">

        <cbc:ID>
            <xsl:value-of select="cbc:ID"/>
        </cbc:ID>

        <xsl:if test="cbc:UUID">

            <cbc:UUID>
                <xsl:value-of select="cbc:UUID"/>
            </cbc:UUID>

        </xsl:if>

        <xsl:if test="cbc:IssueDate">

            <cbc:IssueDate>
                <xsl:value-of select="cbc:IssueDate"/>
            </cbc:IssueDate>

        </xsl:if>

        <xsl:if test="cbc:VersionID">

            <cbc:VersionID>
                <xsl:value-of select="cbc:VersionID"/>
            </cbc:VersionID>

        </xsl:if>

        <xsl:if test="cac:Attachment">

            <cac:Attachment>
                <xsl:apply-templates select="cac:Attachment"/>
            </cac:Attachment>

        </xsl:if>
    </xsl:template>

    <!--  ********************
    */
    */ Templates handling conversion of schemeID
    */ 
    **************************  -->

    <!-- CompanyID inkl. konvertering af schemeID -->
    <xsl:template match="cbc:CompanyID">

        <xsl:attribute name="schemeID">
            <xsl:choose>
                <xsl:when test="@schemeID = 'DK:P'">0096</xsl:when>
                <xsl:when test="@schemeID = 'DK:CVR'">0184</xsl:when>
                <xsl:when test="@schemeID = 'DK:SE'">0198</xsl:when>
                <xsl:otherwise>ZZZ</xsl:otherwise>
            </xsl:choose>
        </xsl:attribute>

        <xsl:value-of select="replace(., '^DK', '')"/>
    </xsl:template>

    <!-- EndPoint og cac:PartyIdentification/cbc:ID inkl. konvertering af schemeID -->
    <xsl:template match="cbc:EndpointID | cac:PartyIdentification/cbc:ID | cac:DeliveryParty/cbc:ID">

        <xsl:copy>
            <xsl:attribute name="schemeID">
                <xsl:choose>
                    <xsl:when test="@schemeID = 'GLN'">0088</xsl:when>
                    <xsl:when test="@schemeID = 'DK:P'">0096</xsl:when>
                    <xsl:when test="@schemeID = 'DK:CVR'">0184</xsl:when>
                    <xsl:when test="@schemeID = 'DK:SE'">0198</xsl:when>
                    <xsl:otherwise>ZZZ</xsl:otherwise>
                </xsl:choose>
            </xsl:attribute>

            <xsl:choose>

                <xsl:when test=".[@schemeID = 'DK:CVR']">
                    <xsl:value-of select="replace(., '^DK', '')"/>
                </xsl:when>

                <xsl:otherwise>
                    <xsl:value-of select="."/>
                </xsl:otherwise>

            </xsl:choose>

        </xsl:copy>

    </xsl:template>

    <!--Contact inkl. opdatering af telefon nummer -->
    <xsl:template match="cac:Contact">

        <cac:Contact>

            <xsl:copy-of select="cbc:ID" copy-namespaces="no"/>

            <xsl:copy-of select="cbc:Name" copy-namespaces="no"/>

            <xsl:comment>Checks and adds '00' to phone# if necessary</xsl:comment>
            <cbc:Telephone>

                <xsl:choose>
                    <xsl:when test="not(matches(cbc:Telephone, '^\+|^00'))">
                        <xsl:value-of select="concat('00', cbc:Telephone)"/>
                    </xsl:when>

                    <xsl:otherwise>
                        <xsl:value-of select="cbc:Telephone"/>
                    </xsl:otherwise>

                </xsl:choose>

            </cbc:Telephone>

            <xsl:copy-of select="cbc:ElectronicMail" copy-namespaces="no"/>
        </cac:Contact>

    </xsl:template>


    <xsl:template name="CompanyID">

        <xsl:param name="value"/>

        <xsl:choose>
            <xsl:when test="string-length($value) = 10 and matches($value, '^[A-Za-z]{2}')">

                <xsl:value-of select="substring($value, 3)"/>
            </xsl:when>

            <xsl:otherwise>

                <xsl:value-of select="$value"/>
            </xsl:otherwise>

        </xsl:choose>
    </xsl:template>

    <!-- Template for Price -->
    <xsl:template match="cac:Price">

        <cac:Price>

            <xsl:apply-templates select="cbc:PriceAmount"/>

            <xsl:apply-templates select="cbc:BaseQuantity"/>

            <xsl:apply-templates select="cac:PriceList"/>

            <xsl:apply-templates select="cac:AllowanceCharge"/>
        </cac:Price>

    </xsl:template>

    <!-- Template for PriceAmount -->
    <xsl:template match="cbc:PriceAmount">

        <cbc:PriceAmount>

            <xsl:attribute name="currencyID">
                <xsl:value-of select="@currencyID"/>
            </xsl:attribute>

            <xsl:value-of select="."/>

        </cbc:PriceAmount>
    </xsl:template>


    <!-- Template for BaseQuantity -->
    <xsl:template match="cbc:BaseQuantity">

        <cbc:BaseQuantity>

            <xsl:attribute name="unitCode">
                <xsl:value-of select="@unitCode"/>
            </xsl:attribute>

            <xsl:value-of select="."/>

        </cbc:BaseQuantity>
    </xsl:template>


    <!-- Template for PriceList -->
    <xsl:template match="cac:PriceList">

        <cac:PriceList>

            <cbc:ID>
                <xsl:value-of select="."/>
            </cbc:ID>

        </cac:PriceList>
    </xsl:template>


    <!-- Template for AllowanceCharge -->
    <xsl:template match="cac:AllowanceCharge">

        <cac:AllowanceCharge>

            <xsl:copy-of select="*[name() != 'cac:TaxCategory']"/>

            <cac:TaxCategory>

                <cbc:ID>
                    <xsl:variable name="idValue" select="normalize-space(cac:TaxCategory/cbc:ID)"/>

                    <xsl:choose>
                        <xsl:when test="$idValue = 'StandardRated'">S</xsl:when>
                        <xsl:when test="$idValue = 'ZeroRated'">Z</xsl:when>
                        <xsl:when test="$idValue = 'ReverseCharge'">AE</xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="$idValue"/>
                        </xsl:otherwise>
                    </xsl:choose>

                </cbc:ID>

                <xsl:copy-of
                    select="cac:TaxCategory/*[not(name() = 'cbc:ID' or name() = 'cac:TaxScheme')]"/>

                <cac:TaxScheme>

                    <xsl:copy-of
                        select="cac:TaxCategory/cac:TaxScheme/*[not(name() = 'cac:JurisdictionRegionAddress')]"/>

                    <xsl:if test="cac:TaxCategory/cac:TaxScheme/cac:JurisdictionRegionAddress">

                        <cac:JurisdictionRegionAddress>

                            <cbc:AddressFormatCode>StructuredRegion</cbc:AddressFormatCode>

                            <cbc:Postbox>1234</cbc:Postbox>

                            <cbc:CityName>Default</cbc:CityName>

                            <cbc:PostalZone>1234</cbc:PostalZone>

                            <xsl:copy-of
                                select="cac:TaxCategory/cac:TaxScheme/cac:JurisdictionRegionAddress/cac:Country"/>

                        </cac:JurisdictionRegionAddress>

                    </xsl:if>

                </cac:TaxScheme>

            </cac:TaxCategory>

        </cac:AllowanceCharge>

    </xsl:template>


    <!-- Template for StandardItemIdentification -->
    <xsl:template match="cac:StandardItemIdentification">

        <xsl:comment>schemeID on StandardItemIdentification/ID is convertes to match ISO 6523 ICD if possible</xsl:comment>
        <cac:StandardItemIdentification>

            <cbc:ID>
                <xsl:attribute name="schemeID">
                    <xsl:choose>
                        <xsl:when test="cbc:ID/@schemeID = 'GTIN'">0160</xsl:when>
                        <xsl:when test="cbc:ID/@schemeID = 'LEI'">0199</xsl:when>
                        <xsl:when test="cbc:ID/@schemeID = 'EBID'">189</xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="cbc:ID/@schemeID"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:attribute>

                <xsl:value-of select="cbc:ID"/>

            </cbc:ID>


        </cac:StandardItemIdentification>
    </xsl:template>


    <!-- BillingReference -->
    <xsl:template match="cac:BillingReference">

        <xsl:element name="{name()}">

            <xsl:if test="not(cac:InvoiceDocumentReference)">

                <xsl:comment>Default InvoiceDocumentReference inserted</xsl:comment>
                <cac:InvoiceDocumentReference>
                    <cbc:ID>Default</cbc:ID>
                </cac:InvoiceDocumentReference>

            </xsl:if>


            <xsl:apply-templates
                select="cac:InvoiceDocumentReference | cac:CreditNoteDocumentReference | cac:ReminderDocumentReference | cac:BillingReferenceLine"
                mode="docRef"/>

        </xsl:element>

    </xsl:template>


    <xsl:template
        match="cac:InvoiceDocumentReference | cac:CreditNoteDocumentReference | cac:ReminderDocumentReference | cac:BillingReferenceLine"
        mode="docRef">

        <xsl:element name="{name()}">

            <xsl:for-each
                select="cbc:ID | cbc:UUID | cbc:IssueDate | cbc:IssueTime | cbc:DocumentDescription">
                <xsl:copy-of select="."/>
            </xsl:for-each>


        </xsl:element>
    </xsl:template>

    <!-- 
        BillingReference end 
    -->

    <xsl:template match="cac:OrderReference" mode="default">
        <xsl:copy>
            <xsl:copy-of select="@* | node()"/>
        </xsl:copy>
    </xsl:template>


    <xsl:template
        match="cac:BuyersItemIdentification | cac:SellersItemIdentification | cac:ManufacturersItemIdentification | cac:CatalogueItemIdentification | cac:AdditionalItemIdentification">

        <xsl:element name="{name()}">

            <cbc:ID>
                <xsl:value-of select="cbc:ID"/>
            </cbc:ID>

        </xsl:element>
    </xsl:template>


    <!-- Ensure correct DocuemntTypeCode - if missing in source a default are inserted  -->
    <xsl:template name="typeCode">

        <xsl:if test="cbc:InvoiceTypeCode[normalize-space()]">
            <cbc:InvoiceTypeCode>
                <xsl:value-of select="cbc:InvoiceTypeCode"/>
            </cbc:InvoiceTypeCode>
        </xsl:if>

        <xsl:if test="cbc:CreditNoteTypeCode[normalize-space()]">
            <cbc:CreditNoteTypeCode>
                <xsl:value-of select="cbc:CreditNoteTypeCode"/>
            </cbc:CreditNoteTypeCode>
        </xsl:if>

        <xsl:if
            test="not(cbc:InvoiceTypeCode[normalize-space()] or cbc:CreditNoteTypeCode[normalize-space()])">
            <xsl:comment>Default InvoiceTypeCode or CreditNoteTypeCode inserted</xsl:comment>
            <xsl:choose>
                <xsl:when test="local-name() = 'Invoice'">
                    <cbc:InvoiceTypeCode>380</cbc:InvoiceTypeCode>
                </xsl:when>
                <xsl:when test="local-name() = 'CreditNote'">
                    <cbc:CreditNoteTypeCode>381</cbc:CreditNoteTypeCode>
                </xsl:when>
            </xsl:choose>
        </xsl:if>
    </xsl:template>


    <!-- Line Referance -->
    <xsl:template
        match="cac:OrderLineReference | cac:DespatchLineReference | cac:ReceiptLineReference">

        <xsl:element name="{name()}">

            <xsl:copy-of select="cbc:LineID" copy-namespaces="no"/>

            <xsl:if test="cac:DocumentReference">
                <cac:DocumentReference>

                    <xsl:copy-of select="cac:DocumentReference/cbc:ID" copy-namespaces="no"/>

                    <xsl:copy-of select="cac:DocumentReference/cbc:UUID" copy-namespaces="no"/>

                    <xsl:copy-of select="cac:DocumentReference/cbc:IssueDate" copy-namespaces="no"/>


                </cac:DocumentReference>

            </xsl:if>
        </xsl:element>
    </xsl:template>


    <!-- Copy elemnent without attributes  -->
    <xsl:template match="cbc:DocumentCurrencyCode | cbc:TaxCurrencyCode | cbc:PricingCurrencyCode">
        <xsl:element name="{name()}">

            <xsl:value-of select="."/>
        </xsl:element>
    </xsl:template>


</xsl:stylesheet>
