<?xml version="1.0" encoding="UTF-8"?>
<!--
    XSLT Transformation for OIOUBL 3 Invoice Response to HTML Structure

    Publisher:         Erhvervsstyrelsen
    Created Date:      2025-04-08
    Last Changed Date: 2025-04-08
    
    Description:       This XSLT transforms OIOUBL 3 Invoice Response XML documents into a semantic HTML structure for display,
                       styled with an external CSS file (OIOUBL30-styling.css). It supports both on-screen viewing and printing,
                       with features like collapsible sections for detailed information. The structure is designed to be modular and
                       compatible with multiple OIOUBL document types where applicable.
    
    Rights:            Licensed under the Creative Commons License; free to use with attribution.
    Contact:           For more information, reach out to Erhvervsstyrelsen.
    
    
    Structure Overview:
    1. Utility Templates: Reusable templates for rendering common elements like party and contact details.
    2. Party Templates: Templates for rendering sender and receiver information.
    3. Document Section Templates: Templates for document parameters, delivery, payment terms, and additional info.
    4. Main Template: Primary template for transforming the root Invoice Response into HTML.
    5. Line Item Details Template: Template for rendering detailed line item information with collapsible sections.
    6. Totals Template: Template for rendering the totals section with formatted currency.
-->

<xsl:stylesheet version="2.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:app="urn:oasis:names:specification:ubl:schema:xsd:ApplicationResponse-2"
    xmlns:cre="urn:oasis:names:specification:ubl:schema:xsd:CreditNote-2" xmlns:cac="urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2"
    xmlns:cbc="urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2" xmlns:xs="http://www.w3.org/2001/XMLSchema" xmlns:my="urn:my:functions"
    exclude-result-prefixes="cac cbc cre app my xs">




    <!--  ********************
    */
    */  Ekstern 
    */
    **************************  -->

    <xsl:variable name="css-content" select="unparsed-text('OIOUBL30-styling.css')" />

    <!-- Output Definition -->
    <xsl:output method="html" encoding="UTF-8" indent="yes" />


    <!--  ********************
    */
    */  Fundctions
    */
    **************************  -->
    <xsl:function name="my:map-schemeID" as="xs:string">
        <xsl:param name="schemeID" as="xs:string" />

        <xsl:sequence select="
                if ($schemeID = '0096') then
                    'DK:P'
                else
                    if ($schemeID = '0184') then
                        'DK:CVR'
                    else
                        if ($schemeID = '0198') then
                            'DK:SE'
                        else
                            if ($schemeID = '0088') then
                                'GLN'
                            else
                                $schemeID" />
    </xsl:function>


    <xsl:function name="my:map-responseCode" as="xs:string">
        <xsl:param name="responseCode" as="xs:string" />

        <xsl:sequence select="
                if ($responseCode = 'AB') then
                    'Message acknowledgement'
                else
                    if ($responseCode = 'AP') then
                        'Accepted'
                    else
                        if ($responseCode = 'RE') then
                            'Rejected'
                        else
                            if ($responseCode = 'IP') then
                                'In process'
                            else
                                if ($responseCode = 'UQ') then
                                    'Under query'
                                else
                                    if ($responseCode = 'CA') then
                                        'Conditionally accepted'
                                    else
                                        if ($responseCode = 'PD') then
                                            'Paid'
                                        else
                                            $responseCode" />
    </xsl:function>



    <xsl:function name="my:map-documentTypeCode" as="xs:string">
        <xsl:param name="documentTypeCode" as="xs:string" />

        <xsl:sequence select="
                if ($documentTypeCode = '380') then
                    'Commercial invoice'
                else
                    if ($documentTypeCode = '381') then
                        'Credit note'
                    else
                        if ($documentTypeCode = '388') then
                            'Tax invoice'
                        else
                            if ($documentTypeCode = '389') then
                                'Self-billed invoice'
                            else
                                if ($documentTypeCode = '393') then
                                    'Factored invoice'
                                else
                                    if ($documentTypeCode = '325') then
                                        'Proforma invoice'
                                    else
                                        $documentTypeCode" />
    </xsl:function>


    <xsl:function name="my:generateXPath" as="xs:string">
        <xsl:param name="pNode" as="item()?"/>
        <xsl:sequence select="
            if (not($pNode) or string($pNode) = '')
                then ''
            else string-join(($pNode/ancestor::*/name(), name($pNode)), '/')
      "/>
    </xsl:function>




    <!-- ========================================================================= -->
    <!-- Utility Templates Section
         Description: Contains reusable templates for rendering common elements like party details, contact information,
                      and address formatting across the document. -->
    <!-- ========================================================================= -->


    <xsl:template name="renderReceiverPartyDetails">
        <xsl:param name="context" select="." />
        <xsl:param name="containerClass" select="'UBLParty'" />

        <xsl:variable name="fields">
            <xsl:if test="$context/cac:PartyLegalEntity/cbc:RegistrationName">
                <xsl:variable name="registrationName" select="$context/cac:PartyLegalEntity/cbc:RegistrationName"/>
                <field title="Partens formelle navn." label="Juridisk navn:" value="{$registrationName}" xpath="{my:generateXPath($registrationName)}" />
            </xsl:if>
            <xsl:if test="$context/cac:PartyIdentification/cbc:ID">
                <xsl:variable name="ID" select="$context/cac:PartyIdentification/cbc:ID"/>
                <xsl:variable name="schemeID" select="$context/cac:PartyIdentification/cbc:ID/@schemeID"/>
                <field title="Supplerende identifikationskode for modtager. Kan eksempelvis være et momsnummer (hvis forskelligt fra CVR-nummeret) eller et P-nummer (til identifikation af den relevante fillial)." label="Supplerende modtageridentifikation:" value="{concat($ID, ' (', my:map-schemeID(string($schemeID)), ')')}" xpath="{my:generateXPath($ID)};{my:generateXPath($schemeID)}" />
            </xsl:if>
            <xsl:if test="$context/cbc:EndpointID">
                <xsl:variable name="endpointID" select="$context/cbc:EndpointID"/>
                <xsl:variable name="schemeID" select="$context/cbc:EndpointID/@schemeID"/>
                <field title="Angiver det Nemhandel endepunkt, f.eks. et CVR eller GLN nummer, hvortil korrespondance vedrørende dette dokument kan stiles." label="Endepunkts-ID:" value="{concat($endpointID, ' (', my:map-schemeID($schemeID), ')')}" xpath="{my:generateXPath($endpointID)};{my:generateXPath($schemeID)}" />
            </xsl:if>
        </xsl:variable>

        <xsl:if test="$fields/field[normalize-space(@value) != '']">
            <div class="{$containerClass}">
                <dl>
                    <xsl:for-each select="$fields/field[normalize-space(@value) != '']">
                        <dt title="{@title}">
                            <xsl:value-of select="@label" />
                        </dt>
                        <dd data-id="{@xpath}">
                            <xsl:value-of select="@value" />
                        </dd>
                    </xsl:for-each>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderSenderPartyDetails">
        <xsl:param name="context" select="." />
        <xsl:param name="containerClass" select="'UBLParty'" />

        <xsl:variable name="fields">
            <xsl:if test="$context/cac:PartyLegalEntity/cbc:RegistrationName">
                <xsl:variable name="registrationName" select="$context/cac:PartyLegalEntity/cbc:RegistrationName"/>
                <field title="Partens formelle navn." label="Juridisk navn:" value="{$registrationName}" xpath="{my:generateXPath($registrationName)}" />
            </xsl:if>
            <xsl:if test="$context/cac:PartyIdentification/cbc:ID">
                <xsl:variable name="ID" select="$context/cac:PartyIdentification/cbc:ID"/>
                <xsl:variable name="schemeID" select="$context/cac:PartyIdentification/cbc:ID/@schemeID"/>
                <field title="Supplerende identifikationskode for afsender. Kan eksempelvis være et momsnummer (hvis forskelligt fra CVR-nummeret) eller et P-nummer (til identifikation af den relevante fillial)." label="Supplerende afsenderidentifikation:" value="{concat($ID, ' (', my:map-schemeID(string($schemeID)), ')')}" xpath="{my:generateXPath($ID)};{my:generateXPath($schemeID)}" />
            </xsl:if>
            <xsl:if test="$context/cbc:EndpointID">
                <xsl:variable name="endpointID" select="$context/cbc:EndpointID"/>
                <xsl:variable name="schemeID" select="$context/cbc:EndpointID/@schemeID"/>
                <field title="Angiver det Nemhandel endepunkt, f.eks. et CVR eller GLN nummer, hvortil korrespondance vedrørende dette dokument kan stiles." label="Endepunkts-ID:" value="{concat($endpointID, ' (', my:map-schemeID($schemeID), ')')}" xpath="{my:generateXPath($endpointID)};{my:generateXPath($schemeID)}" />
            </xsl:if>
        </xsl:variable>

        <xsl:if test="$fields/field[normalize-space(@value) != '']">
            <div class="{$containerClass}">
                <dl>
                    <xsl:for-each select="$fields/field[normalize-space(@value) != '']">
                        <dt title="{@title}">
                            <xsl:value-of select="@label" />
                        </dt>
                        <dd data-id="{@xpath}">
                            <xsl:value-of select="@value" />
                        </dd>
                    </xsl:for-each>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderOriginalIssuerPartyDetails">
        <xsl:param name="context" select="." />
        <xsl:param name="containerClass" select="'UBLParty'" />
        
        <xsl:variable name="fields">
            <xsl:if test="$context/cac:PartyName/cbc:Name">
                <xsl:variable name="name" select="$context/cac:PartyName/cbc:Name"/>
                <field title="Partens virksomhedsnavn" label="Partsnavn:" value="{$name}" xpath="{my:generateXPath($name)}" />
            </xsl:if>
            <xsl:if test="$context/cac:PartyIdentification/cbc:ID">
                <xsl:variable name="ID" select="$context/cac:PartyIdentification/cbc:ID"/>
                <xsl:variable name="schemeID" select="$context/cac:PartyIdentification/cbc:ID/@schemeID"/>
                <field title="Supplerende identifikationskode for parten. Kan eksempelvis være et momsnummer (hvis forskelligt fra CVR-nummeret) eller et P-nummer (til identifikation af den relevante fillial)." label="Supplerende partsidentifikation:" value="{concat($ID, ' (', my:map-schemeID(string($schemeID)), ')')}" xpath="{my:generateXPath($ID)};{my:generateXPath($schemeID)}" />
            </xsl:if>
        
        </xsl:variable>
        
        <xsl:if test="$fields/field[normalize-space(@value) != '']">
            <div class="{$containerClass}">
                <dl>
                    <xsl:for-each select="$fields/field[normalize-space(@value) != '']">
                        <dt title="{@title}">
                            <xsl:value-of select="@label" />
                        </dt>
                        <dd data-id="{@xpath}">
                            <xsl:value-of select="@value" />
                        </dd>
                    </xsl:for-each>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderOriginalRecipientPartyDetails">
        <xsl:param name="context" select="." />
        <xsl:param name="containerClass" select="'UBLParty'" />

        <xsl:variable name="fields">
            <xsl:if test="$context/cac:PartyName/cbc:Name">
                <xsl:variable name="name" select="$context/cac:PartyName/cbc:Name"/>
                <field title="Partens virksomhedsnavn" label="Partsnavn:" value="{$name}" xpath="{my:generateXPath($name)}" />
            </xsl:if>
            <xsl:if test="$context/cac:PartyIdentification/cbc:ID">
                <xsl:variable name="ID" select="$context/cac:PartyIdentification/cbc:ID"/>
                <xsl:variable name="schemeID" select="$context/cac:PartyIdentification/cbc:ID/@schemeID"/>
                <field title="Supplerende identifikationskode for parten. Kan eksempelvis være et momsnummer (hvis forskelligt fra CVR-nummeret) eller et P-nummer (til identifikation af den relevante fillial)." label="Supplerende partsidentifikation:" value="{concat($ID, ' (', my:map-schemeID(string($schemeID)), ')')}" xpath="{my:generateXPath($ID)};{my:generateXPath($schemeID)}" />
            </xsl:if>

        </xsl:variable>

        <xsl:if test="$fields/field[normalize-space(@value) != '']">
            <div class="{$containerClass}">
                <dl>
                    <xsl:for-each select="$fields/field[normalize-space(@value) != '']">
                        <dt title="{@title}">
                            <xsl:value-of select="@label" />
                        </dt>
                        <dd data-id="{@xpath}">
                            <xsl:value-of select="@value" />
                        </dd>
                    </xsl:for-each>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>






    <xsl:template name="renderContactDetails">
        <xsl:param name="context" select="." />
        <xsl:if test="$context/cac:Contact">
            <h3 class="section-title-mini">Kontaktoplysning</h3>
            <div class="UBLContact">
                <dl>
                    <xsl:if test="$context/cac:Contact/cbc:ID">
                        <dt title="Kontaktens ID. Kan være kontaktpersonens medarbejder-ID, eller blot et løbenummer til at sikre unikhed indenfor dokumentet.">Kontaktidentifikation:</dt>
                        <xsl:variable name="ID" select="$context/cac:Contact/cbc:ID"/>
                        <dd data-id="{my:generateXPath($ID)}">
                            <xsl:value-of select="$ID" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:Name">
                        <dt title="Kontaktpunktets navn. Det anbefales at anvende et funktionsnavn.">Navn:</dt>
                        <xsl:variable name="name" select="$context/cac:Contact/cbc:Name"/>
                        <dd data-id="{my:generateXPath($name)}">
                            <xsl:value-of select="$name" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:Telephone">
                        <dt title="Det telefonnummer kontakten kan kontaktes på.">Telefon:</dt>
                        <xsl:variable name="telephone" select="$context/cac:Contact/cbc:Telephone"/>
                        <dd data-id="{my:generateXPath($telephone)}">
                            <xsl:value-of select="$telephone" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:ElectronicMail">
                        <dt title="Kontaktens e-mail adresse.">E-mail:</dt>
                        <xsl:variable name="email" select="$context/cac:Contact/cbc:ElectronicMail"/>
                        <dd data-id="{my:generateXPath($email)}">
                            <xsl:value-of select="$email" />
                        </dd>
                    </xsl:if>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderReceiverContactDetails">
        <xsl:param name="context" select="." />
        <xsl:if test="$context/cac:Contact">
            <h3 title="Oplysninger vedrørende Kontakt." class="section-title-mini">Kontakt</h3>
            <div class="UBLContact">
                <dl>
                    <xsl:if test="$context/cac:Contact/cbc:ID">
                        <dt title="Kontaktens ID. Kan være kontaktpersonens medarbejder-ID, eller blot et løbenummer til at sikre unikhed indenfor dokumentet.">Kontaktidentifikation:</dt>
                        <xsl:variable name="ID" select="$context/cac:Contact/cbc:ID"/>
                        <dd data-id="{my:generateXPath($ID)}">
                            <xsl:value-of select="$ID" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:Name">
                        <dt title="Kontaktpunktets navn. Det anbefales at anvende et funktionsnavn.">Navn:</dt>
                        <xsl:variable name="name" select="$context/cac:Contact/cbc:Name"/>
                        <dd data-id="{my:generateXPath($name)}">
                            <xsl:value-of select="$name" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:Telephone">
                        <dt title="Det telefonnummer kontakten kan kontaktes på.">Telefon:</dt>
                        <xsl:variable name="telephone" select="$context/cac:Contact/cbc:Telephone"/>
                        <dd data-id="{my:generateXPath($telephone)}">
                            <xsl:value-of select="$telephone" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:ElectronicMail">
                        <dt title="Kontaktens e-mail adresse.">E-mail:</dt>
                        <xsl:variable name="email" select="$context/cac:Contact/cbc:ElectronicMail"/>
                        <dd data-id="{my:generateXPath($email)}">
                            <xsl:value-of select="$email" />
                        </dd>
                    </xsl:if>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderSenderContactDetails">
        <xsl:param name="context" select="." />
        <xsl:if test="$context/cac:Contact">
            <h3 title="Kontaktoplysninger for den kommercielt ansvarlige kontaktperson hos afsender." class="section-title-mini">Afsenders kontaktperson</h3>
            <div class="UBLContact">
                <dl>
                    <xsl:if test="$context/cac:Contact/cbc:ID">
                        <dt title="Kontaktens ID. Kan være kontaktpersonens medarbejder-ID, eller blot et løbenummer til at sikre unikhed indenfor dokumentet.">Kontaktidentifikation:</dt>
                        <xsl:variable name="ID" select="$context/cac:Contact/cbc:ID"/>
                        <dd data-id="{my:generateXPath($ID)}">
                            <xsl:value-of select="$ID" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:Name">
                        <dt title="Kontaktpunktets navn. Det anbefales at anvende et funktionsnavn.">Navn:</dt>
                        <xsl:variable name="name" select="$context/cac:Contact/cbc:Name"/>
                        <dd data-id="{my:generateXPath($name)}">
                            <xsl:value-of select="$name" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:Telephone">
                        <dt title="Det telefonnummer kontakten kan kontaktes på.">Telefon:</dt>
                        <xsl:variable name="telephone" select="$context/cac:Contact/cbc:Telephone"/>
                        <dd data-id="{my:generateXPath($telephone)}">
                            <xsl:value-of select="$telephone" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:ElectronicMail">
                        <dt title="Kontaktens e-mail adresse.">E-mail:</dt>
                        <xsl:variable name="email" select="$context/cac:Contact/cbc:ElectronicMail"/>
                        <dd data-id="{my:generateXPath($email)}">
                            <xsl:value-of select="$email" />
                        </dd>
                    </xsl:if>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>



    <!-- ========================================================================= -->
    <!-- Party Templates Section
         Description: Defines templates for rendering party-related information, such as AccountingSupplierParty and AccountingCustomerParty,
                      with conditional logic for invoice or credit note contexts. -->
    <!-- ========================================================================= -->

    <!-- Afsender af svar på modtaget faktura/kreditnota  -->
    <xsl:template match="cac:SenderParty">
        <h2 title="Den part som sender fakturasvaret. Dette er altid den forretningsmæssige modtager af det dokument som besvares (corner 4 i den oprindelige transaktion)." class="section-title">Afsenderpart</h2>
        <xsl:call-template name="renderSenderPartyDetails">
            <xsl:with-param name="containerClass" select="'UBLSender'" />
        </xsl:call-template>
        <xsl:call-template name="renderSenderContactDetails" />
        
          
    </xsl:template>
    
    
    <!-- Faktura/kreditnota modtager-->
    <xsl:template match="cac:DocumentResponse/cac:RecipientParty">
        <h2 title="Den part der modtog den besvarede faktura, hvis dette er forskelligt fra fakturasvarets afsenderpart." class="section-title">Fakturamodtager</h2>
        <xsl:call-template name="renderOriginalRecipientPartyDetails">
            <xsl:with-param name="containerClass" select="'UBLSender'" />
        </xsl:call-template>
        <xsl:call-template name="renderContactDetails" />
    </xsl:template>
    
    
    <!-- Modtager af svar på afsendt faktura/kreditnota  -->
    <xsl:template match="cac:ReceiverParty">
        <h2 title="Den part som modtager fakturasvaret. Dette er altid den forretningsmæssige afsender af det dokument som besvares (corner 1 i den oprindelige transaktion)." class="section-title">Modtagerpart</h2>
        <xsl:call-template name="renderReceiverPartyDetails">
            <xsl:with-param name="containerClass" select="'UBLReceiver'" />
        </xsl:call-template>
        <xsl:call-template name="renderReceiverContactDetails" />
        
    </xsl:template>
    
    <!-- Faktura/kreditnota udsteder-->
    <xsl:template match="cac:DocumentResponse/cac:IssuerParty">
        <h2 title="Den part der udstedte den besvarede faktura, hvis dette er forskelligt fra fakturasvarets modtagerpart." class="section-title">Fakturaudsteder</h2>
        <xsl:call-template name="renderOriginalIssuerPartyDetails">
            <xsl:with-param name="containerClass" select="'UBLReceiver'" />
        </xsl:call-template>
        <xsl:call-template name="renderContactDetails" />
    </xsl:template>
    
    
    <!-- ========================================================================= -->
    <!-- Document Section Templates
         Description: Contains templates for rendering specific sections of the document, such as document parameters, delivery details,
                      payment terms, and additional information. -->
    <!-- ========================================================================= -->

    <xsl:template name="Dokumentparametre">
        <div class="UBLInfo ubl-block">
            <hr />
            <h3 class="section-title-mini">OIOUBL dokumentparametre</h3>
            <dl>
                <dt>Format:</dt>
                <dd>
                    <xsl:call-template name="OIOUBLFormat" />
                </dd>
                <xsl:if test="cbc:CustomizationID">
                    <dt title="Identificerer dokumentet som et OIOUBL dokument">Tilpasnings-ID:</dt>

                    <xsl:variable name="customizationID" select="cbc:CustomizationID"/>
                    <dd data-id="{my:generateXPath($customizationID)}">
                        <xsl:value-of select="$customizationID" />
                    </dd>
                </xsl:if>
                <xsl:if test="cbc:ProfileID">
                    <dt title="Identificerer den OIOUBL profil dokumentet indgår i">ProfilID:</dt>
                    <xsl:variable name="profileID" select="cbc:ProfileID"/>
                    <dd data-id="{my:generateXPath($profileID)}">
                        <xsl:value-of select="$profileID" />
                    </dd>
                </xsl:if>
                <xsl:if test="cbc:CopyIndicator">
                    <dt title="Angiver, om dokumentet, der refereres til, er en kopi (sand) eller originalen (falsk).">Kopi-indikator:</dt>
                    <xsl:variable name="copyIndicator" select="cbc:CopyIndicator"/>
                    <dd data-id="{my:generateXPath($copyIndicator)}">
                        <xsl:value-of select="$copyIndicator" />
                    </dd>
                </xsl:if>
                <dt>Stylesheet version:</dt>
                <dd>3.2.1.22ca3408</dd>
            </dl>
        </div>
    </xsl:template>


    <xsl:template name="OIOUBLFormat">
        <xsl:variable name="customizationID" select="//*[local-name() = 'CustomizationID']" />
        <xsl:value-of select="
                if (contains($customizationID, 'oioubl.dk')) then
                    'OIOUBL 3.0'
                else
                    if (contains($customizationID, 'peppol.eu')) then
                        'Peppol Bis Billing 3.0'
                    else
                    'UNKNOWN'" />
    </xsl:template>




    <xsl:template name="YderligInfo">
        <div class="ubl-block">
            <hr />
            <h2 title="Fritekstfelt. Dette element kan uddybende oplysninger eller instruktioner, der ikke eksplicit er i en anden struktur. Må ikke benyttes til oplysninger der er understøttet af strukturerede felter. I modsætning til de fleste dokumenter er Invoice Response noten afløftningsobligatorisk, da den kan indeholde juridisk betydningsbærende oplysninger." class="section-title-mini">Note</h2>
            <dl>
                <xsl:if test="cbc:Note">
                    <div class="note">
                        <xsl:variable name="note" select="cbc:Note"/>
                        <dt data-id="{my:generateXPath($note)}">
                            <xsl:value-of select="replace($note, '&#10;', '&lt;br/&gt;')" disable-output-escaping="yes" />
                        </dt>
                    </div>
                </xsl:if>
            </dl>
        </div>
    </xsl:template>



    <!-- ========================================================================= -->
    <!-- Main Template Section
         Description: Defines the primary template for transforming the root Invoice or CreditNote into a complete HTML document,
                      including header, body sections, and embedded JavaScript for interactivity. -->
    <!-- ========================================================================= -->

    <xsl:template match="/app:ApplicationResponse">
        <html>
            <head>
                <meta charset="UTF-8" />
                <title>OIOUBL visning</title>


                <style type="text/css">
                    <xsl:value-of select="$css-content" disable-output-escaping="yes" />
                </style>
            </head>
            <body>

                <header class="document-header">
                    <div class="header-container">

                        <div class="header-logo" />

                        <h1 class="document-type">
                            <xsl:choose>
                                <xsl:when test="contains(lower-case(cbc:CustomizationID), 'invoice_response')">
                                    INVOICE RESPONSE
                                </xsl:when>
                                <xsl:otherwise>
                                    UNKNOWEN DOCUMENT TYPE
                                </xsl:otherwise>
                            </xsl:choose>
                      </h1>
                        <dl class="header-details">
                            <xsl:if test="cbc:ID">
                                <div class="header-row">
                                    <dt title="Løbenummer tildelt af afsender, til identifikation af dokumentet.">Løbenummer:</dt>

                                    <xsl:variable name="ID" select="cbc:ID"/>
                                    <dd data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="$ID" />
                                    </dd>
                                </div>
                            </xsl:if>
                            <xsl:if test="cbc:IssueDate">
                                <div class="header-row">
                                    <dt title="Datoen for dokumentets udstedelse. I Nemhandel forventes dokumenter at være udstedt samme dag som de sendes.">Udstedelsesdato:</dt>
                                    <xsl:variable name="issueDate" select="cbc:IssueDate"/>
                                    <dd data-id="{my:generateXPath($issueDate)}">
                                        <xsl:value-of select="$issueDate" />
                                    </dd>
                                </div>
                            </xsl:if>
                            <xsl:if test="cac:DocumentResponse/cac:Response/cbc:EffectiveDate">
                                <div class="header-row">
                                    <dt title="Den dato hvor fakturaen blev kategoriseret i det validerings-trin som er angivet i ResponseCode. Bør være identisk med udstedelsesdatoen, med mindre der er en specifik grund til at fravige den.">Status-dato:</dt>
                                    <xsl:variable name="effectiveDate" select="cac:DocumentResponse/cac:Response/cbc:EffectiveDate"/>
                                    <dd data-id="{my:generateXPath($effectiveDate)}">
                                        <xsl:value-of select="$effectiveDate" />
                                    </dd>
                                </div>
                            </xsl:if>
                            <xsl:if test="cbc:IssueTime">
                                <div class="header-row">
                                    <dt title="Klokkeslættet for dokumentets afsendelse. Er normalt uden juridisk betydning.">Udstedelsestidspunkt:</dt>
                                    <xsl:variable name="issueTime" select="cbc:IssueTime"/>
                                    <dd data-id="{my:generateXPath($issueTime)}">
                                        <xsl:value-of select="$issueTime" />
                                    </dd>
                                </div>
                            </xsl:if>
                            
                        </dl>
                    </div>
                </header>
                <hr class="header-separator" />

                <section class="parties">
                    <div class="party-container">
                        <xsl:apply-templates select="cac:ReceiverParty" />
                        <hr class="short-line" />
                        <xsl:apply-templates select="cac:DocumentResponse/cac:IssuerParty" />
                    </div>
                    <div class="party-container">
                        <xsl:apply-templates select="cac:SenderParty" />
                        <hr class="short-line" />
                        <xsl:apply-templates select="cac:DocumentResponse/cac:RecipientParty" />
                    </div>
                </section>


                <xsl:if test="cbc:Note">
                    <section class="UBLInfo">
                        <xsl:call-template name="YderligInfo" />
                    </section>
                </xsl:if>

                <section class="items">
                    <table class="ItemsTable">
                        <thead>
                            <tr class="UBLLineHeader">
                                <th title="Kode der specificere typen af en faktura. Koden skal være fra OIOUBL UNCL 1001 Invoice Subset" style="width: 20%;">Faktura typekode</th>
                                <th title="Svarkode som angiver hvor fakturaen er i godkendelses-processen, og hvorvidt leverandøren forventes at foretage sig yderligere." style="width: 20%;">Responskode</th>
                                <th title="Fakturanummeret på den faktura der henvises til." style="width: 15%;">Fakturanummer</th>
                                <th title="Den handling som fakturamodtager ønsker at leverandøren foretager sig." style="width: 45%;">Instruks</th>
                             
                            </tr>
                        </thead>
                        <tbody>

                            <tr class="itemRow">
                                <td>
                                    <xsl:variable name="typeCode" select="cac:DocumentResponse/cac:DocumentReference/cbc:DocumentTypeCode"/>
                                    <strong data-id="{my:generateXPath($typeCode)}">
                                        <xsl:value-of
                                            select="concat(my:map-documentTypeCode($typeCode), ' (', $typeCode, ')')"
                                        />
                                    </strong>
                                </td>
                                <td>
                                    <xsl:variable name="responseCode" select="cac:DocumentResponse/cac:Response/cbc:ResponseCode"/>
                                    <strong data-id="{my:generateXPath($responseCode)}">
                                        <xsl:value-of
                                            select="concat(my:map-responseCode($responseCode), ' (', $responseCode, ')')" />
                                    </strong>
                                </td>
                                <td>
                                    <xsl:variable name="ID" select="cac:DocumentResponse/cac:DocumentReference/cbc:ID"/>
                                    <strong data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="$ID" />
                                    </strong>
                                </td>
                                <td>

                                    <xsl:for-each select="cac:DocumentResponse/cac:Response/cac:Status">

                                        <div>
                                            <xsl:attribute name="class">
                                                <xsl:text>Instructiontekst status-block</xsl:text>
                                                <xsl:if test="position() &gt; 1">
                                                    <xsl:text> status-spacing</xsl:text>
                                                </xsl:if>
                                            </xsl:attribute>

                                            <xsl:if test="cbc:StatusReasonCode">
                                                <div>
                                                    <span title="Kode som angiver den ønskede handling." class="Statuslabel">Instrukskode:</span>
                                                    <xsl:variable name="reasonCode" select="cbc:StatusReasonCode"/>
                                                    <span class="Statusværdier" data-id="{my:generateXPath($reasonCode)}">
                                                        <xsl:value-of select="$reasonCode" />
                                                    </span>
                                                </div>
                                            </xsl:if>

                                            <xsl:if test="cbc:StatusReason or cac:Condition/cbc:AttributeID or cac:Condition/cbc:Description">
                                                <hr class="short-line" />
                                                <div class="status-indent">
                                                    <xsl:if test="cbc:StatusReason">
                                                        <div>
                                                            <span title="Beskrivelse af den ønskede handling." class="Statuslabel">Instrukstekst:</span>
                                                            <xsl:variable name="reason" select="cbc:StatusReason"/>
                                                            <span class="Statusværdier" data-id="{my:generateXPath($reason)}">
                                                                <xsl:value-of select="$reason" />
                                                            </span>
                                                        </div>
                                                    </xsl:if>

                                                    <xsl:for-each select="cac:Condition">
                                                        <xsl:if test="cbc:AttributeID or cbc:Description">
                                                            <div class="condition-block condition-spacing">

                                                                <xsl:if test="cbc:AttributeID">
                                                                    <div>
                                                                        <span title="Det EN16931 forretnings-term som korrektionen vedrører." class="Statuslabel">Forretnings-term:</span>

                                                                        <xsl:variable name="attributeID" select="cbc:AttributeID"/>
                                                                        <span class="Statusværdier" data-id="{my:generateXPath($attributeID)}">
                                                                            <xsl:value-of select="$attributeID" />
                                                                        </span>
                                                                    </div>
                                                                </xsl:if>

                                                                <xsl:if test="cbc:Description">
                                                                    <div>
                                                                        <span title="Den værdi fakturamodtager foreslår til det angivne forretnings-term." class="Statuslabel">Data-værdi:</span>
                                                                        <xsl:variable name="description" select="cbc:Description"/>
                                                                        <span class="Statusværdier" data-id="{my:generateXPath($description)}">
                                                                            <xsl:value-of select="$description" />
                                                                        </span>
                                                                    </div>
                                                                </xsl:if>

                                                            </div>
                                                        </xsl:if>
                                                    </xsl:for-each>

                                                </div>
                                            </xsl:if>

                                        </div>
                                    </xsl:for-each>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </section>

                <section class="document-params">
                    <xsl:call-template name="Dokumentparametre" />
                </section>
            </body>
        </html>
    </xsl:template>

</xsl:stylesheet>
