<?xml version="1.0" encoding="UTF-8"?>
<!--
    XSLT Transformation for OIOUBL 3 Invoice Response to HTML Structure

    Publisher:         Erhvervsstyrelsen
    Created Date:      2025-04-08MESSAGE LEVEL RESPONSE
    Last Changed Date: 2025-04-08
    
    Description:       This XSLT transforms OIOUBL 3 Invoice Response XML documents into a semantic HTML structure for display,
                       styled with an external CSS file (OIOUBL30-styling.css). It supports both on-screen viewing and printing,
                       with features like collapsible sections for detailed information. The structure is designed to be modular and
                       compatible with multiple OIOUBL document types where applicable.
    
    Rights:            Licensed under the Creative Commons License; free to use with attribution.
    Contact:           For more information, reach out to Erhvervsstyrelsen.
    
    
    Structure Overview:
    1. Utility Templates: Reusable templates for rendering common elements like party and contact details.
    2. Party Templates: Templates for rendering sender and receiver information.
    3. Document Section Templates: Templates for document parameters, delivery, payment terms, and additional info.
    4. Main Template: Primary template for transforming the root Invoice Response into HTML.
    5. Line Item Details Template: Template for rendering detailed line item information with collapsible sections.
    6. Totals Template: Template for rendering the totals section with formatted currency.
-->

<xsl:stylesheet version="2.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:app="urn:oasis:names:specification:ubl:schema:xsd:ApplicationResponse-2"
    xmlns:cre="urn:oasis:names:specification:ubl:schema:xsd:CreditNote-2" xmlns:cac="urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2"
    xmlns:cbc="urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2" xmlns:xs="http://www.w3.org/2001/XMLSchema" xmlns:my="urn:my:functions"
                xmlns:cl="urn:dk:oioubl:codelist"
                exclude-result-prefixes="cac cbc cre app my xs cl">




    <!--Below function and map is used to lookup codelists-->
    <xsl:variable name="codelist-doc" as="element(mappings)">
                <mappings>
        </mappings>
    </xsl:variable>


    <xsl:function name="cl:lookup" as="xs:string">
        <xsl:param name="xpath" as="xs:string"/>
        <xsl:param name="code" as="xs:anyAtomicType?"/>

        <xsl:variable name="codeStr" select="string($code)"/>


        <xsl:variable name="hit"
                      select="
          $codelist-doc/mapping[
              tokenize(@xpath, '\s*\|\s*') = $xpath
          ]/code[@id = $codeStr]
        "/>

        <xsl:sequence select="
        if (exists($hit))
        then string($hit/@name)
        else $codeStr"/>
    </xsl:function>

    <!--  ********************
    */
    */  Ekstern 
    */
    **************************  -->

    <xsl:variable name="css-content" select="unparsed-text('OIOUBL30-styling.css')" />

    <!-- Output Definition -->
    <xsl:output method="html" encoding="UTF-8" indent="yes" />


    <!--  ********************
    */
    */  Fundctions
    */
    **************************  -->
    <xsl:function name="my:map-schemeID" as="xs:string">
        <xsl:param name="schemeID" as="xs:string" />

        <xsl:sequence select="
                if ($schemeID = '0096') then
                    'DK:P'
                else
                    if ($schemeID = '0184') then
                        'DK:CVR'
                    else
                        if ($schemeID = '0198') then
                            'DK:SE'
                        else
                            if ($schemeID = '0088') then
                                'GLN'
                            else
                                $schemeID" />
    </xsl:function>


    <xsl:function name="my:map-responseCode" as="xs:string">
        <xsl:param name="responseCode" as="xs:string" />

        <xsl:sequence select="
                if ($responseCode = 'AB') then
                    'Message acknowledgement'
                else
                    if ($responseCode = 'AP') then
                        'Accepted'
                    else
                        if ($responseCode = 'RE') then
                            'Rejected'
                        else
                            if ($responseCode = 'IP') then
                                'In process'
                            else
                                if ($responseCode = 'UQ') then
                                    'Under query'
                                else
                                    if ($responseCode = 'CA') then
                                        'Conditionally accepted'
                                    else
                                        if ($responseCode = 'PD') then
                                            'Paid'
                                        else
                                            $responseCode" />
    </xsl:function>



    <xsl:function name="my:map-statusReasonCode" as="xs:string">
        <xsl:param name="statusReasonCode" as="xs:string" />

        <xsl:sequence select="
                if ($statusReasonCode = 'BV') then
                    'Business rule violation, fatal'
                else
                    if ($statusReasonCode = 'BW') then
                        'Business rule violation, warning'
                    else
                        if ($statusReasonCode = 'SV') then
                            'Syntax violation'
                        else
                            if ($statusReasonCode = 'FD') then
                                'Failure of delivery'
                            else
                                $statusReasonCode" />
    </xsl:function>


    <xsl:function name="my:map-documentTypeCode" as="xs:string">
        <xsl:param name="documentTypeCode" as="xs:string" />

        <xsl:sequence select="
                if ($documentTypeCode = '380') then
                    'Commercial invoice'
                else
                    if ($documentTypeCode = '381') then
                        'Credit note'
                    else
                        if ($documentTypeCode = '388') then
                            'Tax invoice'
                        else
                            if ($documentTypeCode = '389') then
                                'Self-billed invoice'
                            else
                                if ($documentTypeCode = '393') then
                                    'Factored invoice'
                                else
                                    if ($documentTypeCode = '325') then
                                        'Proforma invoice'
                                    else
                                        $documentTypeCode" />
    </xsl:function>


    <xsl:function name="my:generateXPath" as="xs:string">
        <xsl:param name="pNode" as="item()?"/>
        <xsl:sequence select="
            if (not($pNode) or string($pNode) = '')
                then ''
            else string-join(($pNode/ancestor::*/name(), name($pNode)), '/')
      "/>
    </xsl:function>



    <!-- ========================================================================= -->
    <!-- Utility Templates Section
         Description: Contains reusable templates for rendering common elements like party details, contact information,
                      and address formatting across the document. -->
    <!-- ========================================================================= -->


    <xsl:template name="renderReceiverPartyDetails">
        <xsl:param name="context" select="." />
        <xsl:param name="containerClass" select="'UBLParty'" />

        <xsl:variable name="fields">
            <xsl:if test="$context/cbc:EndpointID">
                <xsl:variable name="endpointID" select="$context/cbc:EndpointID"/>
                <xsl:variable name="schemeID" select="$context/cbc:EndpointID/@schemeID"/>
                <field title="Angiver det Nemhandel endepunkt, f.eks. et CVR eller GLN nummer, hvortil korrespondance vedrørende dette dokument kan stiles." label="Endepunkts-ID:" value="{concat($endpointID, ' (', my:map-schemeID($schemeID), ')')}" xpath="{my:generateXPath($endpointID)};{my:generateXPath($schemeID)}" />
            </xsl:if>
        </xsl:variable>

        <xsl:if test="$fields/field[normalize-space(@value) != '']">
            <div class="{$containerClass}">
                <dl>
                    <xsl:for-each select="$fields/field[normalize-space(@value) != '']">
                        <dt title="{@title}">
                            <xsl:value-of select="@label" />
                        </dt>
                        <dd data-id="{@xpath}">
                            <xsl:value-of select="@value" />
                        </dd>
                    </xsl:for-each>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderSenderPartyDetails">
        <xsl:param name="context" select="." />
        <xsl:param name="containerClass" select="'UBLParty'" />

        <xsl:variable name="fields">
            <xsl:if test="$context/cbc:EndpointID">
                <xsl:variable name="endpointID" select="$context/cbc:EndpointID"/>
                <xsl:variable name="schemeID" select="$context/cbc:EndpointID/@schemeID"/>
                <field title="Angiver det Nemhandel endepunkt, f.eks. et CVR eller GLN nummer, hvortil korrespondance vedrørende dette dokument kan stiles." label="Endepunkts-ID:" value="{concat($endpointID, ' (', my:map-schemeID($schemeID), ')')}" xpath="{my:generateXPath($endpointID)};{my:generateXPath($schemeID)}" />
            </xsl:if>
        </xsl:variable>

        <xsl:if test="$fields/field[normalize-space(@value) != '']">
            <div class="{$containerClass}">
                <dl>
                    <xsl:for-each select="$fields/field[normalize-space(@value) != '']">
                        <dt title="{@title}">
                            <xsl:value-of select="@label" />
                        </dt>
                        <dd data-id="{@xpath}">
                            <xsl:value-of select="@value" />
                        </dd>
                    </xsl:for-each>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderContactDetails">
        <xsl:param name="context" select="." />
        <xsl:if test="$context/cac:Contact">
            <h3 class="section-title-mini">Kontaktoplysning</h3>
            <div class="UBLContact">
                <dl>
                    <xsl:if test="$context/cac:Contact/cbc:ID">
                        <dt title="Kontaktens ID. Kan være kontaktpersonens medarbejder-ID, eller blot et løbenummer til at sikre unikhed indenfor dokumentet.">Kontaktidentifikation:</dt>
                        <xsl:variable name="ID" select="$context/cac:Contact/cbc:ID"/>
                        <dd data-id="{my:generateXPath($ID)}">
                            <xsl:value-of select="$ID" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:Name">
                        <dt title="Kontaktpunktets navn. Det anbefales at anvende et funktionsnavn.">Navn:</dt>
                        <xsl:variable name="name" select="$context/cac:Contact/cbc:Name"/>
                        <dd data-id="{my:generateXPath($name)}">
                            <xsl:value-of select="$name" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:Telephone">
                        <dt title="Det telefonnummer kontakten kan kontaktes på.">Telefon:</dt>
                        <xsl:variable name="telephone" select="$context/cac:Contact/cbc:Telephone"/>
                        <dd data-id="{my:generateXPath($telephone)}">
                            <xsl:value-of select="$telephone" />
                        </dd>
                    </xsl:if>
                    <xsl:if test="$context/cac:Contact/cbc:ElectronicMail">
                        <dt title="Kontaktens e-mail adresse.">E-mail:</dt>
                        <xsl:variable name="email" select="$context/cac:Contact/cbc:ElectronicMail"/>
                        <dd data-id="{my:generateXPath($email)}">
                            <xsl:value-of select="$email" />
                        </dd>
                    </xsl:if>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <!-- ========================================================================= -->
    <!-- Party Templates Section
         Description: Defines templates for rendering party-related information, such as AccountingSupplierParty and AccountingCustomerParty,
                      with conditional logic for invoice or credit note contexts. -->
    <!-- ========================================================================= -->

    <xsl:template match="cac:SenderParty">
        <h2 class="section-title">Afsender</h2>
        <xsl:call-template name="renderSenderPartyDetails">
            <xsl:with-param name="containerClass" select="'UBLSender'" />
        </xsl:call-template>
        <xsl:call-template name="renderContactDetails" />
    </xsl:template>

    <xsl:template match="cac:ReceiverParty">
        <h2 class="section-title">Modtager</h2>
        <xsl:call-template name="renderReceiverPartyDetails">
            <xsl:with-param name="containerClass" select="'UBLReceiver'" />
        </xsl:call-template>
        <xsl:call-template name="renderContactDetails" />
    </xsl:template>

    <!-- ========================================================================= -->
    <!-- Document Section Templates
         Description: Contains templates for rendering specific sections of the document, such as document parameters, delivery details,
                      payment terms, and additional information. -->
    <!-- ========================================================================= -->

    <xsl:template name="Dokumentparametre">
        <div class="UBLInfo ubl-block">
            <hr />
            <h3 class="section-title-mini">OIOUBL dokumentparametre</h3>
            <dl>
                <dt>Format:</dt>
                <dd>
                    <xsl:call-template name="OIOUBLFormat" />
                </dd>
                <xsl:if test="cbc:CustomizationID">
                    <dt DkDescription="Identificerer dokumentet som et OIOUBL dokument">Tilpasnings-ID:</dt>
                    <xsl:variable name="customizationID" select="cbc:CustomizationID"/>
                    <dd data-id="{my:generateXPath($customizationID)}">
                        <xsl:value-of select="$customizationID" />
                    </dd>
                </xsl:if>
                <xsl:if test="cbc:ProfileID">
                    <dt DkDescription="Identificerer den OIOUBL profil dokumentet indgår i.">ProfilID:</dt>
                    <xsl:variable name="profileID" select="cbc:ProfileID"/>
                    <dd data-id="{my:generateXPath($profileID)}">
                        <xsl:value-of select="$profileID" />
                    </dd>
                </xsl:if>
                <xsl:if test="cbc:CopyIndicator">
                    <dt DkDescription="Angiver, om dokumentet, der refereres til, er en kopi (sand) eller originalen (falsk).">Kopi-indikator:</dt>
                    <xsl:variable name="copyIndicator" select="cbc:CopyIndicator"/>
                    <dd data-id="{my:generateXPath($copyIndicator)}">
                        <xsl:value-of select="$copyIndicator" />
                    </dd>
                </xsl:if>
                <dt>Stylesheet version:</dt>
                <dd>3.2.3.0e92060a</dd>
            </dl>
        </div>
    </xsl:template>


    <xsl:template name="OIOUBLFormat">
        <xsl:variable name="customizationID" select="//*[local-name() = 'CustomizationID']" />
        <xsl:value-of select="
                if (contains($customizationID, 'oioubl.dk')) then
                    'OIOUBL 3.0'
                else
                    if (contains($customizationID, 'peppol.eu')) then
                        'Peppol Bis Billing 3.0'
                    else
                    'UNKNOWN'" />
    </xsl:template>




    <xsl:template name="YderligInfo">
        <div class="ubl-block">
            <hr />
            <h2 class="section-title-mini">Yderligere oplysninger</h2>
            <dl>
                <xsl:if test="cbc:Note">
                    <div class="note">
                        <xsl:variable name="note" select="cbc:Note"/>
                        <dt data-id="{my:generateXPath($note)}">
                            <xsl:value-of select="replace($note, '&#10;', '&lt;br/&gt;')" disable-output-escaping="yes" />
                        </dt>
                    </div>
                </xsl:if>
            </dl>

        </div>
    </xsl:template>



    <!-- ========================================================================= -->
    <!-- Main Template Section
         Description: Defines the primary template for transforming the root Invoice or CreditNote into a complete HTML document,
                      including header, body sections, and embedded JavaScript for interactivity. -->
    <!-- ========================================================================= -->

    <xsl:template match="/app:ApplicationResponse">
        <html>
            <head>
                <meta charset="UTF-8" />
                <title>OIOUBL visning</title>


                <style type="text/css">
                    <xsl:value-of select="$css-content" disable-output-escaping="yes" />
                </style>
            </head>
            <body>

                <header class="document-header">
                    <div class="header-container">

                        <div class="header-logo" />

                        <h1 class="document-type">
                            <xsl:choose>
                                <xsl:when test="contains(lower-case(cbc:CustomizationID), 'message_level_response')"> MESSAGE LEVEL RESPONSE </xsl:when>
                                <xsl:otherwise> UNKNOWEN DOCUMENT TYPE </xsl:otherwise>
                            </xsl:choose>
                        </h1>
                        <dl class="header-details">
                            <xsl:if test="cbc:ID">
                                <div class="header-row">
                                    <dt title="Løbenummer tildelt af afsender, til identifikation af dokumentet.">Løbenummer:</dt>
                                    <xsl:variable name="ID" select="cbc:ID"/>
                                    <dd data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="$ID" />
                                    </dd>
                                </div>
                            </xsl:if>
                            <xsl:if test="cbc:IssueDate">
                                <div class="header-row">
                                    <dt title="Datoen for dokumentets udstedelse. I Nemhandel forventes dokumenter at være udstedt samme dag som de sendes.">Udstedelsesdato:</dt>
                                    <xsl:variable name="issueDate" select="cbc:IssueDate"/>
                                    <dd data-id="{my:generateXPath($issueDate)}">
                                        <xsl:value-of select="cbc:IssueDate" />
                                    </dd>
                                </div>
                            </xsl:if>
                            <xsl:if test="cbc:IssueTime">
                                <div class="header-row">
                                    <dt title="Klokkeslættet for dokumentets afsendelse. Er normalt uden juridisk betydning.">Udstedelsestidspunkt:</dt>
                                    <xsl:variable name="issueTime" select="cbc:IssueTime"/>
                                    <dd data-id="{my:generateXPath($issueTime)}">
                                        <xsl:value-of select="$issueTime" />
                                    </dd>
                                </div>
                            </xsl:if>
                            <xsl:if test="cac:DocumentResponse/cac:DocumentReference/cbc:VersionID">
                                <div class="header-row">
                                    <dt title="Den version af dokumentet svaret omhandler. (F.eks. katalogversion.)">Dokumentets versionsnummer:</dt>
                                    <xsl:variable name="versionID" select="cac:DocumentResponse/cac:DocumentReference/cbc:VersionID"/>
                                    <dd data-id="{my:generateXPath($versionID)}">
                                        <xsl:value-of select="$versionID" />
                                    </dd>
                                </div>
                            </xsl:if>
                            


                        </dl>
                    </div>
                </header>
                <hr class="header-separator" />

                <section class="parties">
                    <div class="party-container">
                        <xsl:apply-templates select="cac:ReceiverParty" />
                    </div>
                    <div class="party-container">
                        <xsl:apply-templates select="cac:SenderParty" />
                    </div>
                </section>


                <xsl:if test="cbc:Note">
                    <section class="UBLInfo">
                        <xsl:call-template name="YderligInfo" />
                    </section>
                </xsl:if>

                <section class="items">
                    <table class="ItemsTable">
                        <thead>
                            <tr class="UBLLineHeader">
                                <th title="Dokumentets type angivet som en kode." style="width: 20%;">Dokumenttype-kode</th>
                                <th title="Svarkode som angiver om dokumentet har fejlet syntaks-verifikation, bestået syntaks-verifikation, eller er modtaget af den endelige modtager." style="width: 20%;">Responskode</th>
                                <th title="Den unikke instance-identifier som er knyttet til den forsendelse som besvares. Findes i den header som blev anvendt til dokument-udvekslingen." style="width: 15%;">Dokumentets kuvertnummer</th>
                                <th title="Begrundelsen for den status dokumentet er tildelt i ResponseCode. Obligatorisk hvis dokumentet er afvist. Hvis dokumentet afvises med en valideringsfejl bør dette felt inkludere en fejlkode." style="width: 45%;">Svartekst</th>

                            </tr>
                        </thead>
                        <tbody>

                            <tr class="itemRow">
                                <td>
                                    <xsl:variable name="typeCode" select="cac:DocumentResponse/cac:DocumentReference/cbc:DocumentTypeCode"/>
                                    <strong data-id="{my:generateXPath($typeCode)}">
                                        <xsl:value-of
                                            select="concat(my:map-documentTypeCode($typeCode), ' (', $typeCode, ')')"
                                         />
                                    </strong>
                                </td>
                                <td>
                                    <xsl:variable name="reasonCode" select="cac:DocumentResponse/cac:Response/cbc:ResponseCode"/>
                                    <strong data-id="{my:generateXPath($reasonCode)}">
                                        <xsl:value-of
                                            select="concat(my:map-responseCode($reasonCode), ' (', $reasonCode, ')')" />
                                    </strong>
                                </td>
                                <td>
                                    <xsl:variable name="ID" select="cac:DocumentResponse/cac:DocumentReference/cbc:ID"/>
                                    <strong data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="$ID" />
                                    </strong>
                                </td>
                                <td>
                                    <xsl:variable name="responseDescription" select="cac:DocumentResponse/cac:Response/cbc:Description"/>
                                    <strong>
                                        <xsl:value-of
                                            select="concat(my:map-responseCode($responseDescription), ' (', $responseDescription, ')')" />
                                    </strong>

                                  
                                    <xsl:for-each select="cac:DocumentResponse/cac:LineResponse">
                                        <hr />
                                        <div class=" svarTekstMLR {
                                            concat('status-indent ',
                                            if (position() mod 2 = 1)
                                            then 'oddrowMRL'
                                            else 'evenrowMLR')
                                            }">

                                            <div>
                                                <xsl:attribute name="class">
                                                    <xsl:text>Instructiontekst status-block</xsl:text>

                                                    <xsl:text> status-spacing</xsl:text>
                                                </xsl:attribute>

                                                <xsl:if test="cac:LineReference">
                                                    <div>
                                                        <span title="Identifikation for Linjen i referencedokumentet." class="Statuslabel">LinjeID:</span>
                                                        <xsl:variable name="lineID" select="cac:LineReference/cbc:LineID"/>
                                                        <span class="Statusværdier" data-id="{my:generateXPath($lineID)}">
                                                            <b><xsl:value-of select="$lineID" /></b>
                                                        </span>
                                                    </div>
                                                </xsl:if>


                                                <xsl:for-each select="cac:Response">

                                                    <hr class="short-line" />

                                                    <xsl:if test="cbc:Description">
                                                        <div>
                                                            <span title="Beskrivelsen af linjesvaret. Skal angives på engelsk." class="Statuslabel">Beskrivelse:</span>
                                                            <xsl:variable name="description" select="cbc:Description"/>
                                                            <span class="Statusværdier" data-id="{my:generateXPath($description)}">
                                                                <xsl:value-of select="$description" />
                                                            </span>
                                                        </div>
                                                    </xsl:if>

                                                    <xsl:if test="cbc:ResponseCode or cac:Status/cbc:StatusReasonCode">
                                                        <div class="condition-block condition-spacing">

                                                            <xsl:if test="cbc:ResponseCode">
                                                                <div>
                                                                    <span title="Kode for linjesvaret." class="Statuslabel">Svarkode:</span>
                                                                    <xsl:variable name="responseCode" select="cbc:ResponseCode"/>
                                                                    <span class="Statusværdier" data-id="{my:generateXPath($responseCode)}">
                                                                        <xsl:value-of select="concat(my:map-responseCode($responseCode), ' (', $responseCode, ')')" />
                                                                    </span>
                                                                </div>
                                                            </xsl:if>

                                                            <xsl:if test="cac:Status/cbc:StatusReasonCode">
                                                                <div>
                                                                    <span title="Kode der angiver typen af teknisk fejl i linjen." class="Statuslabel">Fejltypekode:</span>
                                                                    <xsl:variable name="reasonCode" select="cac:Status/cbc:StatusReasonCode"/>
                                                                    <span class="Statusværdier" data-id="{my:generateXPath($reasonCode)}">
                                                                        <xsl:value-of
                                                                            select="concat(my:map-statusReasonCode($reasonCode), ' (', $reasonCode, ')')" />
                                                                    </span>
                                                                </div>
                                                            </xsl:if>
                                                        </div>
                                                    </xsl:if>
                                                </xsl:for-each>

                                        </div>

                                        </div>
                                    </xsl:for-each>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </section>

                <section class="document-params">
                    <xsl:call-template name="Dokumentparametre" />
                </section>
            </body>
        </html>
    </xsl:template>

</xsl:stylesheet>
