<?xml version="1.0" encoding="UTF-8"?>
<!--
    XSLT Transformation for OIOUBL 3 Invoice and Credit Note to HTML Structure

    Publisher:         Erhvervsstyrelsen
    Created Date:      2025-03-30
    Last Changed Date: 2025-03-10
    
    Description:       This XSLT transforms OIOUBL 3 Invoice and Credit Note XML documents into a semantic HTML structure for display,
                       styled with an external CSS file (OIOUBL30-styling.css). It supports both on-screen viewing and printing,
                       with features like collapsible sections for detailed information. The structure is designed to be modular and
                       compatible with multiple OIOUBL document types where applicable.
    
    Rights:            Licensed under the Creative Commons License; free to use with attribution.
    Contact:           For more information, reach out to Erhvervsstyrelsen.
    
    
    Structure Overview:
    1. Utility Templates: Reusable templates for rendering common elements like party and contact details.
    2. Party Templates: Templates for rendering sender and receiver information.
    3. Document Section Templates: Templates for document parameters, delivery, payment terms, and additional info.
    4. Main Template: Primary template for transforming the root Invoice or CreditNote into HTML.
    5. Line Item Details Template: Template for rendering detailed line item information with collapsible sections.
    6. Totals Template: Template for rendering the totals section with formatted currency.
-->

<xsl:stylesheet version="2.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:cre="urn:oasis:names:specification:ubl:schema:xsd:CreditNote-2" xmlns:cac="urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2"
    xmlns:cbc="urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2" xmlns:xs="http://www.w3.org/2001/XMLSchema" xmlns:my="urn:my:functions"
                xmlns:cl="urn:dk:oioubl:codelist"
                exclude-result-prefixes="cac cbc cre my xs cl">




    <!--Below function and map is used to lookup codelists-->

    <xsl:variable name="codelist-doc" as="element(mappings)">
                <mappings>
        </mappings>
    </xsl:variable>


    <xsl:function name="cl:lookup" as="xs:string">
        <xsl:param name="xpath" as="xs:string"/>
        <xsl:param name="code" as="xs:anyAtomicType?"/>

        <xsl:variable name="codeStr" select="string($code)"/>


        <xsl:variable name="hit"
                      select="
          $codelist-doc/mapping[
              tokenize(@xpath, '\s*\|\s*') = $xpath
          ]/code[@id = $codeStr]
        "/>

        <xsl:sequence select="
        if (exists($hit))
        then string($hit/@name)
        else $codeStr"/>
    </xsl:function>


    <!--  ********************
    */
    */  Ekstern 
    */
    **************************  -->

    <xsl:variable name="css-content" select="unparsed-text('OIOUBL30-styling.css')" />

    <!-- Output Definition -->
    <xsl:output method="html" encoding="UTF-8" indent="yes" />


    <!--  ********************
    */
    */  Fundctions
    */
    **************************  -->
    <xsl:function name="my:map-schemeID" as="xs:string">
        <xsl:param name="schemeID" as="xs:string" />

        <xsl:sequence select="
                if ($schemeID = '0096') then
                    'DK:P'
                else
                    if ($schemeID = '0184') then
                        'DK:CVR'
                    else
                        if ($schemeID = '0198') then
                            'DK:SE'
                        else
                            if ($schemeID = '0088') then
                                'GLN'
                            else
                                $schemeID" />
    </xsl:function>



    <xsl:function name="my:map-taxCategory" as="xs:string">
        <xsl:param name="taxCategory" as="xs:string" />

        <xsl:sequence select="
                if ($taxCategory = 'AE') then
                    'Vat Reverse Charge'
                else
                    if ($taxCategory = 'E') then
                        'Exempt from Tax'
                    else
                        if ($taxCategory = 'S') then
                            'Standard rate'
                        else
                            if ($taxCategory = 'Z') then
                                'Zero rated goods'
                            else
                                $taxCategory" />
    </xsl:function>


    <xsl:function name="my:map-gebyrRabat" as="xs:string">
        <xsl:param name="gebyrRabat" as="xs:string" />

        <xsl:sequence select="
                if ($gebyrRabat = 'false') then
                    'Rabat'
                else
                    if ($gebyrRabat = 'true') then
                        'Gebyr'
                    else
                        $gebyrRabat" />
    </xsl:function>


    <xsl:function name="my:map-momsKategori" as="xs:string">
        <xsl:param name="momsKategori" as="xs:string" />

        <xsl:sequence select="
                if ($momsKategori = 'StandardRated') then
                    'S'
                else
                    if ($momsKategori = 'ZeroRated') then
                        'Z'
                    else
                        if ($momsKategori = 'ReverseCharge') then
                            'AE'
                        else
                            $momsKategori" />
    </xsl:function>


    <xsl:function name="my:generateXPath" as="xs:string">
        <xsl:param name="pNode" as="item()?"/>
        <xsl:sequence select="
            if (not($pNode) or string($pNode) = '')
                then ''
            else string-join(($pNode/ancestor::*/name(), name($pNode)), '/')
      "/>
    </xsl:function>



    <!-- ========================================================================= -->
    <!-- Utility Templates Section
         Description: Contains reusable templates for rendering common elements like party details, contact information,
                      and address formatting across the document. -->
    <!-- ========================================================================= -->


    <xsl:template name="renderSupplierPartyDetails">
        <xsl:param name="context" select="." />
        <xsl:param name="containerClass" select="'UBLParty'" />

        <xsl:variable name="fields">

            <field label="Partsnavn" title="Kreditors virksomhedsnavn." value="{cac:Party/cac:PartyName/cbc:Name}" xpath="{my:generateXPath(cac:Party/cac:PartyName/cbc:Name)}" />
            <field label="Gadenavn" title="Navnet på den gade adressen er tilknyttet." value="{concat(cac:Party/cac:PostalAddress/cbc:StreetName, ' ', cac:Party/cac:PostalAddress/cbc:BuildingNumber)}" xpath="{my:generateXPath(cac:Party/cac:PostalAddress/cbc:StreetName)};{my:generateXPath(cac:Party/cac:PostalAddress/cbc:BuildingNumber)}" />
            <field label="Postnummer./Postdistrikt" title="Adressens postnummer.&#013;Navnet på det postdistrikt adressen ligger i." value="{concat(cac:Party/cac:PostalAddress/cbc:PostalZone, ' ', cac:Party/cac:PostalAddress/cbc:CityName)}" xpath="{my:generateXPath(cac:Party/cac:PostalAddress/cbc:PostalZone)};{my:generateXPath(cac:Party/cac:PostalAddress/cbc:CityName)}" />
            <field label="Landekode" title="Tocifret FN-landekode." value="{cl:lookup('CreditNote/cac:AccountingSupplierParty/cac:Party/cac:PostalAddress/cac:Country/cbc:IdentificationCode', cac:Party/cac:PostalAddress/cac:Country/cbc:IdentificationCode)}" xpath="{my:generateXPath(cac:Party/cac:PostalAddress/cac:Country/cbc:IdentificationCode)}" />
        </xsl:variable>

        <xsl:variable name="additionalFields">
            <xsl:if test="cac:Party/cac:PartyLegalEntity/cbc:CompanyID">
                <field label="Juridisk enhed:" value="{concat(cac:Party/cac:PartyLegalEntity/cbc:CompanyID, ' (', my:map-schemeID(cac:Party/cac:PartyLegalEntity/cbc:CompanyID/@schemeID), ')')}" xpath="{my:generateXPath(cac:Party/cac:PartyLegalEntity/cbc:CompanyID)};{my:generateXPath(cac:Party/cac:PartyLegalEntity/cbc:CompanyID/@schemeID)}" />
            </xsl:if>
            <xsl:if test="cac:Party/cac:PartyTaxScheme/cbc:CompanyID">
                <field label="Momsnummer:" value="{concat(cac:Party/cac:PartyTaxScheme/cbc:CompanyID, ' (', my:map-schemeID(string(cac:Party/cac:PartyTaxScheme/cbc:CompanyID/@schemeID)), ')')}" xpath="{my:generateXPath(cac:Party/cac:PartyTaxScheme/cbc:CompanyID)};{my:generateXPath(cac:Party/cac:PartyTaxScheme/cbc:CompanyID/@schemeID)}" />
            </xsl:if>
            <field label="EndepunktID:" value="{concat(cac:Party/cbc:EndpointID, ' (', my:map-schemeID(cac:Party/cbc:EndpointID/@schemeID), ')')}" xpath="{my:generateXPath(cac:Party/cbc:EndpointID)};{my:generateXPath(cac:Party/cbc:EndpointID/@schemeID)}" />
        </xsl:variable>

        <xsl:if test="$fields/field[normalize-space(@value) != ''] or $additionalFields/field[normalize-space(@value) != '']">
            <div class="{$containerClass}">
                <!-- Faste felter, der altid vises -->
                <dl>
                    <xsl:for-each select="$fields/field[normalize-space(@value) != '']">
                        <dt title="{@title}">
                            <xsl:value-of select="@label" />
                        </dt>
                        <dd data-id="{@xpath}">
                            <xsl:value-of select="@value" />
                        </dd>
                    </xsl:for-each>
                </dl>

                <!-- Sammenklappelige felter -->
                <xsl:if test="$additionalFields/field[normalize-space(@value) != '']">
                    <div class="UBLAddRefLine">
                        <div class="section-divider">
                            <span class="toggle-button" onclick="toggleAllInfo(this)">Yderligere oplysninger</span>
                        </div>
                        <div class="toggle-info" style="display: none;">
                            <dl>
                                <xsl:for-each select="$additionalFields/field[normalize-space(@value) != '']">
                                    <dt>
                                        <xsl:value-of select="@label" />
                                    </dt>
                                    <dd data-id="{@xpath}">
                                        <xsl:value-of select="@value" />
                                    </dd>
                                </xsl:for-each>
                            </dl>
                        </div>
                    </div>
                </xsl:if>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderCustomerPartyDetails">
        <xsl:param name="context" select="." />
        <xsl:param name="containerClass" select="'UBLParty'" />

        <xsl:variable name="fields">

            <field label="Partsnavn" title="Kreditors virksomhedsnavn." value="{cac:Party/cac:PartyName/cbc:Name}" xpath="{my:generateXPath(cac:Party/cac:PartyName/cbc:Name)}" />
            <field label="Gadenavn" title="Navnet på den gade adressen er tilknyttet." value="{concat(cac:Party/cac:PostalAddress/cbc:StreetName, ' ', cac:Party/cac:PostalAddress/cbc:BuildingNumber)}" xpath="{my:generateXPath(cac:Party/cac:PostalAddress/cbc:StreetName)};{my:generateXPath(cac:Party/cac:PostalAddress/cbc:BuildingNumber)}" />
            <field label="Postnummer./Postdistrikt" title="Adressens postnummer.&#013;Navnet på det postdistrikt adressen ligger i." value="{concat(cac:Party/cac:PostalAddress/cbc:PostalZone, ' ', cac:Party/cac:PostalAddress/cbc:CityName)}" xpath="{my:generateXPath(cac:Party/cac:PostalAddress/cbc:PostalZone)};{my:generateXPath(cac:Party/cac:PostalAddress/cbc:CityName)}" />
            <field label="Landekode" title="Tocifret FN-landekode." value="{cl:lookup('CreditNote/cac:AccountingCustomerParty/cac:Party/cac:PostalAddress/cac:Country/cbc:IdentificationCode', cac:Party/cac:PostalAddress/cac:Country/cbc:IdentificationCode)}" xpath="{my:generateXPath(cac:Party/cac:PostalAddress/cac:Country/cbc:IdentificationCode)}" />
        </xsl:variable>

        <xsl:variable name="additionalFields">
            <xsl:if test="cac:Party/cac:PartyLegalEntity/cbc:CompanyID">
                <field label="Juridisk enhed:" value="{concat(cac:Party/cac:PartyLegalEntity/cbc:CompanyID, ' (', my:map-schemeID(cac:Party/cac:PartyLegalEntity/cbc:CompanyID/@schemeID), ')')}" xpath="{my:generateXPath(cac:Party/cac:PartyLegalEntity/cbc:CompanyID)};{my:generateXPath(cac:Party/cac:PartyLegalEntity/cbc:CompanyID/@schemeID)}" />
            </xsl:if>
            <xsl:if test="cac:Party/cac:PartyTaxScheme/cbc:CompanyID">
                <field label="Momsnummer:" value="{concat(cac:Party/cac:PartyTaxScheme/cbc:CompanyID, ' (', my:map-schemeID(string(cac:Party/cac:PartyTaxScheme/cbc:CompanyID/@schemeID)), ')')}" xpath="{my:generateXPath(cac:Party/cac:PartyTaxScheme/cbc:CompanyID)};{my:generateXPath(cac:Party/cac:PartyTaxScheme/cbc:CompanyID/@schemeID)}" />
            </xsl:if>
            <field label="EndepunktID:" value="{concat(cac:Party/cbc:EndpointID, ' (', my:map-schemeID(cac:Party/cbc:EndpointID/@schemeID), ')')}" xpath="{my:generateXPath(cac:Party/cbc:EndpointID)};{my:generateXPath(cac:Party/cbc:EndpointID/@schemeID)}" />
        </xsl:variable>

        <xsl:if test="$fields/field[normalize-space(@value) != ''] or $additionalFields/field[normalize-space(@value) != '']">
            <div class="{$containerClass}">
                <!-- Faste felter, der altid vises -->
                <dl>
                    <xsl:for-each select="$fields/field[normalize-space(@value) != '']">
                        <dt title="{@title}">
                            <xsl:value-of select="@label" />
                        </dt>
                        <dd data-id="{@xpath}">
                            <xsl:value-of select="@value" />
                        </dd>
                    </xsl:for-each>
                </dl>

                <!-- Sammenklappelige felter -->
                <xsl:if test="$additionalFields/field[normalize-space(@value) != '']">
                    <div class="UBLAddRefLine">
                        <div class="section-divider">
                            <span class="toggle-button" onclick="toggleAllInfo(this)">Yderligere oplysninger</span>
                        </div>
                        <div class="toggle-info" style="display: none;">
                            <dl>
                                <xsl:for-each select="$additionalFields/field[normalize-space(@value) != '']">
                                    <dt>
                                        <xsl:value-of select="@label" />
                                    </dt>
                                    <dd data-id="{@xpath}">
                                        <xsl:value-of select="@value" />
                                    </dd>
                                </xsl:for-each>
                            </dl>
                        </div>
                    </div>
                </xsl:if>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderCustomerContactDetails">
        <xsl:param name="context" select="." />
        <xsl:if test="$context/cac:Party/cac:Contact">
            <h3 title="Kontaktoplysninger for den kommercielt ansvarlige kontaktperson hos debitor." class="section-title-mini">Debitors kontaktperson</h3>
            <div class="UBLContact">

                <dl>
                    <dt title="Kontaktens ID. Kan være kontaktpersonens medarbejder-ID, eller blot et løbenummer til at sikre unikhed indenfor dokumentet.">Kontaktidentifikation:</dt>
                    <xsl:variable name="contactId" select="$context/cac:Party/cac:Contact/cbc:ID"/>
                    <dd data-id="{my:generateXPath($contactId)}">
                        <xsl:value-of select="$contactId" />
                    </dd>
                    <dt title="Kontaktpunktets navn. Det anbefales at anvende et funktionsnavn.">Navn:</dt>
                    <xsl:variable name="name" select="$context/cac:Party/cac:Contact/cbc:Name"/>
                    <dd data-id="{my:generateXPath($name)}">
                        <xsl:value-of select="$name" />
                    </dd>
                    <dt title="Det telefonnummer kontakten kan kontaktes på.">Telefon:</dt>
                    <xsl:variable name="telephone" select="$context/cac:Party/cac:Contact/cbc:Telephone"/>
                    <dd data-id="{my:generateXPath($telephone)}">
                        <xsl:value-of select="$telephone" />
                    </dd>
                    <dt title="Kontaktens e-mail adresse.">E-mail:</dt>
                    <xsl:variable name="electronicMail" select="$context/cac:Party/cac:Contact/cbc:ElectronicMail"/>
                    <dd data-id="{my:generateXPath($electronicMail)}">
                        <xsl:value-of select="$electronicMail" />
                    </dd>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="renderSupplierContactDetails">
        <xsl:param name="context" select="." />
        <xsl:if test="$context/cac:Party/cac:Contact">
            <h3 title="Kontaktoplysninger for den kommercielt ansvarlige kontaktperson hos kreditor." class="section-title-mini">Kreditors kontaktperson</h3>
            <div class="UBLContact">

                <dl>
                    <dt title="Kontaktens ID. Kan være kontaktpersonens medarbejder-ID, eller blot et løbenummer til at sikre unikhed indenfor dokumentet.">Kontaktidentifikation:</dt>
                    <xsl:variable name="contactId" select="$context/cac:Party/cac:Contact/cbc:ID"/>
                    <dd data-id="{my:generateXPath($contactId)}">
                        <xsl:value-of select="$contactId" />
                    </dd>
                    <dt title="Kontaktpunktets navn. Det anbefales at anvende et funktionsnavn.">Navn:</dt>
                    <xsl:variable name="name" select="$context/cac:Party/cac:Contact/cbc:Name"/>
                    <dd data-id="{my:generateXPath($name)}">
                        <xsl:value-of select="$name" />
                    </dd>
                    <dt title="Det telefonnummer kontakten kan kontaktes på.">Telefon:</dt>
                    <xsl:variable name="telephone" select="$context/cac:Party/cac:Contact/cbc:Telephone"/>
                    <dd data-id="{my:generateXPath($telephone)}">
                        <xsl:value-of select="$telephone" />
                    </dd>
                    <dt title="Kontaktens e-mail adresse.">E-mail:</dt>
                    <xsl:variable name="electronicMail" select="$context/cac:Party/cac:Contact/cbc:ElectronicMail"/>
                    <dd data-id="{my:generateXPath($electronicMail)}">
                        <xsl:value-of select="$electronicMail" />
                    </dd>
                </dl>
            </div>
        </xsl:if>
    </xsl:template>

    <xsl:template name="getAddress">
        <xsl:param name="xpathPrefix" />
        <xsl:value-of select="
                normalize-space(concat(
                $xpathPrefix/cac:Address/cbc:StreetName, ' ',
                $xpathPrefix/cac:Address/cbc:BuildingNumber, ', ',
                $xpathPrefix/cac:Address/cbc:PostalZone, ' ',
                $xpathPrefix/cac:Address/cbc:CityName, ', ',
                $xpathPrefix/cac:Address/cac:Country/cbc:IdentificationCode))" />
    </xsl:template>

    <!-- ========================================================================= -->
    <!-- Party Templates Section
         Description: Defines templates for rendering party-related information, such as AccountingSupplierParty and AccountingCustomerParty,
                      with conditional logic for credit note contexts. -->
    <!-- ========================================================================= -->

    <xsl:template match="cac:AccountingSupplierParty">
        <h2 class="section-title">
            <xsl:choose>
                <xsl:when test="local-name(..) = 'Invoice'">Fakturaafsender</xsl:when>
                <xsl:otherwise>Kreditnotaafsender</xsl:otherwise>
            </xsl:choose>
        </h2>
        <xsl:call-template name="renderSupplierPartyDetails">
            <xsl:with-param name="containerClass" select="'UBLPayeeParty'" />
        </xsl:call-template>
        <xsl:call-template name="renderSupplierContactDetails" />
    </xsl:template>

    <xsl:template match="cac:AccountingCustomerParty">
        <h2 class="section-title">
            <xsl:choose>
                <xsl:when test="local-name(..) = 'Invoice'">Fakturamodtager</xsl:when>
                <xsl:otherwise>Kreditnotamodtager</xsl:otherwise>
            </xsl:choose>
        </h2>
        <xsl:call-template name="renderCustomerPartyDetails">
            <xsl:with-param name="containerClass" select="'UBLBuyerCustomerParty'" />
        </xsl:call-template>
        <xsl:call-template name="renderCustomerContactDetails" />
    </xsl:template>

    <!-- ========================================================================= -->
    <!-- Document Section Templates
         Description: Contains templates for rendering specific sections of the document, such as document parameters, delivery details,
                      payment terms, and additional information. -->
    <!-- ========================================================================= -->

    <xsl:template name="Dokumentparametre">
        <div class="UBLInfo ubl-block">
            <hr />
            <h3 class="section-title-mini">OIOUBL dokumentparametre</h3>
            <dl>
                <dt>Format:</dt>
                <dd>
                    <xsl:call-template name="OIOUBLFormat" />
                </dd>
                <xsl:if test="cbc:CustomizationID">
                    <dt title="Identificerer dokumentet som et OIOUBL dokument">Tilpasnings-ID:</dt>
                    <xsl:variable name="customizationID" select="cbc:CustomizationID"/>
                    <dd data-id="{my:generateXPath($customizationID)}">
                        <xsl:value-of select="$customizationID" />
                    </dd>
                </xsl:if>
                <xsl:if test="cbc:ProfileID">
                    <dt title="Identificerer hvilken OIOUBL forretningsproces dokumentet indgår i.">ProfilID:</dt>
                    <xsl:variable name="profileID" select="cbc:ProfileID"/>
                    <dd data-id="{my:generateXPath($profileID)}">
                        <xsl:value-of select="$profileID" />
                    </dd>
                </xsl:if>
                <xsl:if test="cbc:CopyIndicator">
                    <dt title="Indikerer om et dokument er en kopi (true) eller original (false). Hvis dette felt ikke er inkluderet, antages det at det aktuelle dokument er originalen.">Kopi-indikator:</dt>
                    <xsl:variable name="copyIndicator" select="cbc:CopyIndicator"/>
                    <dd data-id="{my:generateXPath($copyIndicator)}">
                        <xsl:value-of select="$copyIndicator" />
                    </dd>
                </xsl:if>
                <dt>Stylesheet version:</dt>
                <dd>3.2.4.b8030f18</dd>
            </dl>
        </div>
    </xsl:template>


    <xsl:template name="OIOUBLFormat">
        <xsl:variable name="customizationID" select="//*[local-name() = 'CustomizationID']" />
        <xsl:value-of select="
                if (contains($customizationID, 'oioubl.dk')) then
                    'OIOUBL 3.0'
                else
                    if (contains($customizationID, 'peppol.eu')) then
                        'Peppol Bis Billing 3.0'
                    else
                        'OIOUBL 2.1'" />
    </xsl:template>


    <xsl:template name="Levering">
        <div class="ubl-block">
            <hr />
            <h2 class="section-title-mini">Levering</h2>
            <dl>
                <xsl:if test="cac:Delivery/cbc:ActualDeliveryDate">
                    <dt title="Den faktiske Leveringsdato for (del-)leverancen.">Faktisk leveringsdato:</dt>
                    <xsl:variable name="actualDeliveryDate" select="cac:Delivery/cbc:ActualDeliveryDate"/>
                    <dd data-id="{my:generateXPath($actualDeliveryDate)}">
                        <xsl:value-of select="$actualDeliveryDate" />
                    </dd>
                </xsl:if>
                <xsl:if test="cac:Delivery/cac:DeliveryLocation/cac:Address">
                    <dt title="Oplysninger vedrørende lokationens adresse.">Adresse:</dt>
                    <xsl:variable name="streetName" select="cac:Delivery/cac:DeliveryLocation/cac:Address/cbc:StreetName"/>
                    <xsl:variable name="postalZone" select="cac:Delivery/cac:DeliveryLocation/cac:Address/cbc:PostalZone"/>
                    <xsl:variable name="cityName" select="cac:Delivery/cac:DeliveryLocation/cac:Address/cbc:CityName"/>
                    <xsl:variable name="identificationCode" select="cac:Delivery/cac:DeliveryLocation/cac:Address/cac:Country/cbc:IdentificationCode"/>
                    <dd data-id="{my:generateXPath($streetName)};{my:generateXPath($postalZone)};{my:generateXPath($cityName)};{my:generateXPath($identificationCode)}">
                        <xsl:value-of select="
                                concat(
                                $streetName, ', ',
                                $postalZone, ' ',
                                $cityName, ', ',
                                $identificationCode)" />
                    </dd>
                </xsl:if>

            </dl>
        </div>
    </xsl:template>


    <xsl:template name="YderligInfo">
        <div class="ubl-block">
            <hr />
            <h2 class="section-title-mini">Yderligere oplysninger</h2>
            <dl>
                <xsl:if test="cbc:Note">
                    <div class="note">
                        <xsl:variable name="note" select="cbc:Note"/>
                        <dt data-id="{my:generateXPath($note)}">
                            <xsl:value-of select="replace($note, '&#10;', '&lt;br/&gt;')" disable-output-escaping="yes" />
                        </dt>
                    </div>
                </xsl:if>
            </dl>

        </div>
    </xsl:template>


    <xsl:template match="cac:PaymentTerms">
        <div class="UBLInfo ubl-block">
            <xsl:if test="position() = 1">
                <hr />
                <h3 title="Oplysninger om fakturaens betalingsbetingelser" class="section-title-mini">Betalingsbetingelser</h3>
            </xsl:if>
            <dl>
                <xsl:if test="cbc:Note">
                    <dt title="Fritekstbeskrivelse af de gældende betalingsbetingelser. Hvor de strukturerede datafelter er udfyldt er disse styrende.">Betalingsbetingelses-detaljer:</dt>
                    <xsl:variable name="note" select="cbc:Note"/>
                    <dd data-id="{my:generateXPath($note)}">
                        <xsl:value-of select="$note" />
                    </dd>
                </xsl:if>
                <xsl:if test="cbc:PenaltySurchargePercent">
                    <dt title="Det procentvise gebyr for betaling efter disse betalingsbetingelser, f.eks. hvor det er aftalt at debitor kan udskyde betalingen imod et gebyr.">Gebyr, procent:</dt>
                    <xsl:variable name="penaltySurchargePercent" select="cbc:PenaltySurchargePercent"/>
                    <dd data-id="{my:generateXPath($penaltySurchargePercent)}">
                        <xsl:value-of select="$penaltySurchargePercent" />
                    </dd>
                </xsl:if>
                <xsl:if test="cbc:PenaltyAmount">
                    <dt title="Det samlede gebyr for betaling efter disse betalingsbetingelser, f.eks. hvor det er aftalt at debitor kan udskyde betalingen imod et gebyr.">Gebyr, total:</dt>
                    <xsl:variable name="penaltyAmount" select="cbc:PenaltyAmount"/>
                    <dd data-id="{my:generateXPath($penaltyAmount)}">
                        <xsl:value-of select="format-number(($penaltyAmount), '###,###,##0.00')" />
                    </dd>
                </xsl:if>
            </dl>
        </div>
    </xsl:template>

    <xsl:template match="cac:PaymentMeans">
        <div class="UBLInfo ubl-block">
            <xsl:if test="position() = 1">
                <hr />
                <h3 title="Oplysninger til brug ved betaling." class="section-title-mini">Betalingsdetaljer</h3>
            </xsl:if>
            <dl>
                <dt title="Kode der identificerer betalingsformen (f.eks. Bankoverførsel eller girokort).">Betalingsform:</dt>
                <dd data-id="{my:generateXPath(cbc:PaymentMeansCode)}">
                    <xsl:choose>
                        <xsl:when test="cbc:PaymentMeansCode = '1'">Betalingsmåde ikke fastlagt (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '10'">Kontant (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '31'">International bankoverførsel (IBAN) (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '42'">Kontooverførsel (DK:BANK) (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '58'">BetalingsService, international (IBAN) (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '48'">Bank (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '49'">Direct Debit, herunder Betalings- eller leverandørservice (IBAN eller DK:BANK) (<xsl:value-of select="cbc:PaymentMeansCode"
                             />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '50'">Giroindbetalingskort (DK:GIRO) (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '58'">Bankoverførsel, national og international (IBAN) (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '59'">BetalingsService, international (IBAN) (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '93'">Fælles indbetalingskort - FIK kort (DK:FIK) (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '57'">BetalingsService, international (IBAN) (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>
                        <xsl:when test="cbc:PaymentMeansCode = '97'">NemKonto (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:when>

                        <xsl:otherwise>Ukendt betalingsmetode (<xsl:value-of select="cbc:PaymentMeansCode" />)</xsl:otherwise>
                    </xsl:choose>
                </dd>
                <xsl:if test="cbc:PaymentChannelCode">
                    <dt title="Kode der anvendes hvor PaymentMeansCode ikke entydigt identificerer betalingsformen (f.eks. til at skelne imellem IBAN og andre typer bankoverførsel).">Detaljeret betalingsform:</dt>
                    <xsl:variable name="paymentChannelCode" select="cbc:PaymentChannelCode"/>
                    <dd data-id="{my:generateXPath($paymentChannelCode)}">
                        <xsl:value-of select="$paymentChannelCode" />
                    </dd>
                </xsl:if>
                <xsl:if test="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cac:FinancialInstitution/cbc:Name">
                    <dt title="Pengeinstituttets navn.">Navn:</dt>
                    <xsl:variable name="name" select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cac:FinancialInstitution/cbc:Name"/>
                    <dd data-id="{my:generateXPath($name)}">
                        <xsl:value-of select="$name" />
                    </dd>
                </xsl:if>
                <xsl:if test="cbc:InstructionID or cbc:PaymentID or cac:PayeeFinancialAccount or cac:PayerFinancialAccount or cac:CreditAccount">
                    <dt>Betalingsoplysninger:</dt>
                    <dd>
                        <xsl:call-template name="printFinancialAccountDetails" />
                    </dd>
                </xsl:if>
                <xsl:if test="cac:CardAccount/cbc:NetworkID">
                    <dt title="Obligatorisk felt i UBL, som ikke har nogen semantisk betydning for de i OIOUBL og EN16931 understøttede betalingskort.">Netværks-ID:</dt>
                    <xsl:variable name="networkID" select="cac:CardAccount/cbc:NetworkID"/>
                    <dd data-id="{my:generateXPath($networkID)}">
                        <xsl:value-of select="$networkID" />
                    </dd>
                </xsl:if>
                <xsl:if test="cac:CardAccount/cbc:HolderName">
                    <dt title="Navnet på den person som kortet er udstedt til.">Kortholders navn:</dt>
                    <xsl:variable name="holderName" select="cac:CardAccount/cbc:HolderName"/>
                    <dd data-id="{my:generateXPath($holderName)}">
                        <xsl:value-of select="$holderName" />
                    </dd>
                </xsl:if>


                <xsl:if test="cbc:InstructionNote">
                    <dt title="Betalingsinstruks til brug ved korttype 01">Lang betalingsinstruks:</dt>
                    <xsl:variable name="instructionNote" select="cbc:InstructionNote"/>
                    <dd data-id="{my:generateXPath($instructionNote)}">
                        <xsl:value-of select="$instructionNote" />
                    </dd>
                </xsl:if>
                <dt>
                    <xsl:if test="position() != last()">
                        <xsl:attribute name="class">border-gradient</xsl:attribute>
                    </xsl:if>
                </dt>
            </dl>
        </div>
    </xsl:template>

    <xsl:template name="printFinancialAccountDetails">
        <xsl:choose>
            <xsl:when test="cbc:PaymentMeansCode = '93'">

                <xsl:variable name="paymentID" select="cbc:PaymentID"/>
                <xsl:variable name="instructionID" select="cbc:InstructionID"/>
                <xsl:variable name="accountID" select="cac:CreditAccount/cbc:AccountID"/>

                <span class="bold" data-id="{my:generateXPath($paymentID)};{my:generateXPath($instructionID)};{my:generateXPath($accountID)}">FIK:</span>
                <xsl:value-of select="concat(' &gt; ', $paymentID, ' ', $instructionID, ' +', $accountID, ' &lt;')" />

            </xsl:when>
            <xsl:when test="cbc:PaymentMeansCode = '31'">

                <xsl:variable name="institutionID" select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cac:FinancialInstitution/cbc:ID"/>
                <xsl:variable name="branchID" select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cbc:ID"/>
                <xsl:variable name="accountID" select="cac:PayeeFinancialAccount/cbc:ID"/>

                <span class="bold" data-id="{my:generateXPath($institutionID)};{my:generateXPath($branchID)};{my:generateXPath($accountID)}">SWIFT:</span>
                <xsl:value-of select="
                        concat(' ',
                        $institutionID, ' ', $branchID, ' ',
                        'Kontonr: ', $accountID)" />

            </xsl:when>
            <xsl:when test="cbc:PaymentMeansCode = '42'">

                <xsl:variable name="institutionID" select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cac:FinancialInstitution/cbc:ID"/>
                <xsl:variable name="branchID" select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cbc:ID"/>
                <xsl:variable name="accountID" select="cac:PayeeFinancialAccount/cbc:ID"/>

                <span class="bold" data-id="{my:generateXPath($institutionID | $branchID)};{my:generateXPath($accountID)}">Reg:</span>
                <xsl:value-of select="
                        concat(' ',
                        ($institutionID | $branchID), ' ',
                        'Kontonr: ', $accountID)" />

            </xsl:when>
            <xsl:when test="cbc:PaymentMeansCode = '48'">

                <xsl:variable name="accountNumberID" select="cac:CardAccount/cbc:PrimaryAccountNumberID"/>
                <xsl:variable name="paymentID" select="cbc:PaymentID"/>

                <span class="bold" data-id="{my:generateXPath($accountNumberID)};{my:generateXPath($paymentID)}">Konto :</span>
                <xsl:value-of select="concat(' &gt; ', $accountNumberID, ' ', $paymentID, ' &lt;')" />

            </xsl:when>
            <xsl:when test="cbc:PaymentMeansCode = '49'">

                <xsl:variable name="instructionID" select="cbc:InstructionID"/>

                <span class="bold" data-id="{my:generateXPath($instructionID)}">Konto :</span>
                <xsl:value-of select="$instructionID" />

            </xsl:when>
            <xsl:when test="cbc:PaymentMeansCode = '50'">

                <xsl:variable name="paymentID" select="cbc:PaymentID"/>
                <xsl:variable name="instructionID" select="cbc:InstructionID"/>
                <xsl:variable name="accountID" select="cac:PayeeFinancialAccount/cbc:ID"/>

                <span class="bold" data-id="{my:generateXPath($paymentID)};{my:generateXPath($instructionID)};{my:generateXPath($accountID)}">GIRO: </span>
                <xsl:value-of select="concat(' &gt; ', $paymentID, ' ', $instructionID, ' ', $accountID, ' &lt;')" />

            </xsl:when>
            <xsl:when test="cbc:PaymentMeansCode = '58'">

                <xsl:if test="cac:PayeeFinancialAccount/cbc:PaymentNote">
                    <xsl:variable name="paymentNote" select="cac:PayeeFinancialAccount/cbc:PaymentNote"/>
                    <span class="bold" data-id="{my:generateXPath($paymentNote)}">Note: </span><xsl:value-of select="$paymentNote" /> -
                </xsl:if>

                <xsl:variable name="institutionID" select="cac:PayeeFinancialAccount/cac:FinancialInstitutionBranch/cbc:ID"/>
                <xsl:variable name="accountID" select="cac:PayeeFinancialAccount/cbc:ID"/>

                <span class="bold" data-id="{my:generateXPath($institutionID)};{my:generateXPath($accountID)}">Konto: </span>
                <xsl:value-of select="
                        normalize-space(concat(
                        $institutionID, ' ',
                        $accountID))" />

            </xsl:when>
            <xsl:when test="cbc:PaymentMeansCode = '59'">

                <xsl:variable name="paymentID" select="cbc:PaymentID"/>
                <xsl:variable name="paymentMandateID" select="cac:PaymentMandate/cbc:ID"/>
                <xsl:variable name="accountID" select="cac:PaymentMandate/cac:PayerFinancialAccount/cbc:ID"/>

                <span class="bold" data-id="{my:generateXPath($paymentID)};{my:generateXPath($paymentMandateID)};{my:generateXPath($accountID)}">Konto: </span>
                <xsl:value-of select="concat(' &gt; ', $paymentID, ' ', $paymentMandateID, ' ', $accountID, ' &lt;')" />
            </xsl:when>
            <xsl:when test="cbc:PaymentMeansCode = '97'">
                <span class="bold">NemKonto</span>
            </xsl:when>
            <xsl:otherwise>
                <span class="payment-info">Ingen konto-oplysninger tilgængelige</span>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>


    <!-- ========================= -->

    <!-- Template: Moms Total      -->

    <!-- ========================= -->

    <xsl:template name="render-tax-total">
        <xsl:for-each-group select="cac:TaxTotal/cac:TaxSubtotal" group-by="floor((position() - 1) div 2)">
            <div class="TaxTotalRow">
                <xsl:for-each select="current-group()">
                    <div class="UBLInfo ubl-block TaxTotalCell">
                        <Table>
                            <xsl:if test="cac:TaxCategory/cbc:ID">
                                <tr class="TaxTotal">
                                    <td colspan="2" class="TaxAndTotalText" title="Momskoden for momskategorien.">Momskode:</td>

                                    <xsl:variable name="taxcategoryID" select="cac:TaxCategory/cbc:ID"/>
                                    <td colspan="3" data-id="{my:generateXPath($taxcategoryID)}" class="TaxCodeText">
                                        <xsl:value-of select="concat($taxcategoryID, ' (', my:map-taxCategory($taxcategoryID), ')')" />
                                    </td>
                                </tr>
                            </xsl:if>

                            <xsl:if test="cac:TaxCategory/cbc:TaxExemptionReasonCode">
                                <tr class="TaxTotal">
                                    <td colspan="2" class="TaxAndTotalText" title="Grunden til momsfritagelse, angivet som kode.">Momsfritagelseskode:</td>

                                    <xsl:variable name="taxExemptionReasonCode" select="cac:TaxCategory/cbc:TaxExemptionReasonCode"/>
                                    <td colspan="3" data-id="{my:generateXPath($taxExemptionReasonCode)}" class="TaxCodeText">
                                        <xsl:value-of select="$taxExemptionReasonCode" />
                                    </td>
                                </tr>
                            </xsl:if>

                            <xsl:if test="cac:TaxCategory/cbc:TaxExemptionReason">
                                <tr class="TaxTotal">
                                    <td colspan="2" class="TaxAndTotalText" title="Grunden til momsfritagelse.">Momsfritagelsesgrund:</td>

                                    <xsl:variable name="TaxExemptionReason" select="cac:TaxCategory/cbc:TaxExemptionReason"/>
                                    <td colspan="3" data-id="{my:generateXPath($TaxExemptionReason)}" class="TaxCodeText">
                                        <xsl:value-of select="$TaxExemptionReason" />
                                    </td>
                                </tr>
                            </xsl:if>

                            <tr class="TaxTotalBorder">
                                <td colspan="5" style="border-top: 2px solid black;" />
                            </tr>

                            <xsl:if test="cbc:TaxableAmount">
                                <tr class="TaxTotal">
                                    <td colspan="2" class="TaxAndTotalText" title="Det (netto)beløb som er omfattet af den angivne TaxCategory.">Momspligtigt beløb:</td>

                                    <xsl:variable name="amount" select="cbc:TaxableAmount" />
                                    <xsl:variable name="currencyID" select="cbc:TaxableAmount/@currencyID"/>
                                    <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($currencyID)}">
                                        <xsl:value-of select="$currencyID" />
                                        <xsl:value-of select="cl:lookup('CreditNote/cac:TaxTotal/cac:TaxSubtotal/cbc:TaxableAmount/@currencyID', $currencyID)" />
                                    </td>
                                    <td class="TaxAndTotalValue numeric" data-id="{my:generateXPath($amount)}">
                                        <xsl:value-of select="format-number(number($amount), '###,###,##0.00')" />
                                    </td>
                                    <td>
                                    </td>
                                </tr>
                            </xsl:if>

                            <xsl:if test="cac:TaxCategory/cbc:Percent">
                                <tr class="TaxTotal">
                                    <td colspan="2" class="TaxAndTotalText" title="Momskategoriens moms-sats, i procent.">Momssats:</td>

                                    <xsl:variable name="percent" select="cac:TaxCategory/cbc:Percent" />
                                    <td class="TaxAndTotalCurrency">

                                    </td>
                                    <td class="TaxAndTotalValue numeric" style="padding-right:32.5px;" data-id="{my:generateXPath($percent)}">
                                        <xsl:value-of select="number($percent)" />
                                    </td>
                                    <td class="TaxAndTotalSuffix">
                                        %
                                    </td>
                                </tr>
                            </xsl:if>

                            <tr class="TaxTotalBorder">
                                <td colspan="5" style="border-top: 2px solid black;" />
                            </tr>

                            <xsl:if test="cbc:TaxAmount">
                                <tr class="TaxTotal">
                                    <td colspan="2" class="TaxAndTotalText" title="Den samlede moms-subtotal.">Momsbeløb:</td>

                                    <xsl:variable name="amount" select="cbc:TaxAmount" />
                                    <xsl:variable name="currencyID" select="cbc:TaxAmount/@currencyID"/>
                                    <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($currencyID)}">
                                        <xsl:value-of select="cl:lookup('CreditNote/cac:TaxTotal/cac:TaxSubtotal/cbc:TaxAmount/@currencyID', $currencyID)" />
                                    </td>
                                    <td class="TaxAndTotalValue numeric" data-id="{my:generateXPath($amount)}">
                                        <xsl:value-of select="format-number(number($amount), '###,###,##0.00')" />
                                    </td>
                                    <td>
                                    </td>
                                </tr>
                            </xsl:if>
                        </Table>
                    </div>
                </xsl:for-each>
            </div>
        </xsl:for-each-group>
    </xsl:template>




    <!-- ============================== -->



    <!-- Template: Rabat & Afgifter      -->



    <!-- ============================== -->



    <xsl:template name="render-allowance-charge">
        <xsl:for-each-group select="cac:AllowanceCharge" group-by="floor((position() - 1) div 2)">
            <div class="AllowanceChargeRow">
                <xsl:for-each select="current-group()">
                    <div class="UBLInfo ubl-block AllowanceChargeCell">
                        <dl>
                            <xsl:if test="cbc:ChargeIndicator">
                                <dt title="TRUE hvis instansen repræsenterer et gebyr eller tillæg, der skal lægges til dokumentsummen. FALSE hvis instansen repræsenterer en rabat, der skal fradrages dokumentsummen.">Gebyrindikator:</dt>
                                <dd>
                                    <xsl:variable name="chargeIndicator" select="cbc:ChargeIndicator"/>
                                    <span class="AmountBoxHeader" data-id="{my:generateXPath($chargeIndicator)}">
                                        <xsl:value-of select="concat(my:map-gebyrRabat($chargeIndicator), ' (', $chargeIndicator, ')')" />
                                    </span>
                                </dd>
                            </xsl:if>

                            <dt>
                                <hr class="TaxSeparator-md" />
                            </dt>
                            <dd />

                            <xsl:if test="cbc:AllowanceChargeReasonCode">
                                <dt title="Kode til identifikation af gebyret eller rabattens karakter.">Rabat/gebyr-kode:</dt>
                                <dd>
                                    <xsl:variable name="allowanceChargeReasonCode" select="cbc:AllowanceChargeReasonCode"/>
                                    <span class="AmountBoxHeader" data-id="{my:generateXPath($allowanceChargeReasonCode)}">
                                        <xsl:value-of select="cl:lookup('CreditNote/cac:AllowanceCharge/cbc:AllowanceChargeReasonCode', $allowanceChargeReasonCode)" />
                                    </span>
                                </dd>
                            </xsl:if>



                            <xsl:if test="cbc:Amount">
                                <dt title="Rabattens eller gebyrets absolutte størrelse.">Rabat/gebyr beløb:</dt>
                                <dd>
                                    <xsl:variable name="amount" select="cbc:Amount"/>
                                    <xsl:variable name="currencyID" select="cbc:Amount/@currencyID"/>
                                    <span class="AmountBoxHeader" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                                        <xsl:value-of select="concat(format-number(($amount), '###,###,##0.00'), ' ', cl:lookup('CreditNote/cac:AllowanceCharge/cbc:Amount/@currencyID', $currencyID))" />
                                    </span>
                                </dd>
                            </xsl:if>

                            <xsl:if test="cbc:BaseAmount">
                                <dt title="Det grundbeløb som en forholdsmæssig rabat eller gebyr beregnes udfra.">Grundbeløb:</dt>
                                <dd>
                                    <xsl:variable name="baseAmount" select="cbc:BaseAmount"/>
                                    <xsl:variable name="currencyID" select="cbc:BaseAmount/@currencyID"/>
                                    <span class="AmountBoxHeader" data-id="{my:generateXPath($baseAmount)};{my:generateXPath($currencyID)}">
                                        <xsl:value-of select="concat(format-number(($baseAmount), '###,###,##0.00'), ' ', cl:lookup('CreditNote/cac:AllowanceCharge/cbc:BaseAmount/@currencyID', $currencyID))" />
                                    </span>
                                </dd>
                            </xsl:if>

                            <xsl:if test="cbc:MultiplierFactorNumeric">
                                <dt title="Den faktor hvormed BaseAmount skal ganges for at finde den samlede rabat eller det samlede gebyr.&#013;Bemærk at EN16931 her afviger fra OASIS: EN16931 anvender felttypen Percentage her, hvor OASIS UBL anvender felttypen Numeric. OIOUBL 3 følger EN16931, og beregner derfor AllowanceCharge.Amount som BaseAmount x MultiplierFactorNumeric / 100.">Rabat/gebyr-faktor:</dt>
                                <dd>
                                    <xsl:variable name="multiplierFactorNumeric" select="cbc:MultiplierFactorNumeric"/>

                                    <span class="AmountBoxHeader" data-id="{my:generateXPath($multiplierFactorNumeric)}">
                                        <xsl:value-of select="$multiplierFactorNumeric" />
                                    </span>
                                </dd>
                            </xsl:if>

                            <xsl:if test="cbc:AllowanceChargeReason">
                                <dt title="Kort beskrivelse af grunden til rabatten eller gebyret.">Rabat/gebyr begrundelse:</dt>
                                <dd>
                                    <xsl:variable name="allowanceChargeReason" select="cbc:AllowanceChargeReason"/>
                                    <span class="multiline-text" data-id="{my:generateXPath($allowanceChargeReason)}">
                                        <xsl:value-of select="$allowanceChargeReason" />
                                    </span>
                                </dd>
                            </xsl:if>



                            <xsl:for-each select="cac:TaxCategory">

                                <dt>
                                    <hr class="TaxSeparator-sm" />
                                </dt>
                                <dd />

                                <xsl:if test="cbc:ID">
                                    <dt title="Momskoden for momskategorien.">Momskode:</dt>
                                    <dd>
                                        <xsl:variable name="ID" select="cbc:ID"/>
                                        <span class="AmountBoxHeader" data-id="{my:generateXPath($ID)}">
                                            <xsl:value-of select="concat($ID, ' (', my:map-taxCategory($ID), ')')" />
                                        </span>
                                    </dd>
                                </xsl:if>


                                <xsl:if test="cbc:Percent">
                                    <dt title="Momskategoriens moms-sats, i procent.">Momssats:</dt>
                                    <dd>
                                        <xsl:variable name="percent" select="cbc:Percent"/>
                                        <span class="AmountBoxHeader" data-id="{my:generateXPath($percent)}">
                                            <xsl:value-of select="concat($percent, ' %')" />
                                        </span>
                                    </dd>
                                </xsl:if>

                                <xsl:if test="cac:TaxScheme/cbc:ID">
                                    <dt title="Identificerer typen af skat eller afgift. P.t. understøtter EN16931 alene moms (''VAT'').">Afgiftstypekode:</dt>
                                    <dd>
                                        <xsl:variable name="taxSchemeID" select="cac:TaxScheme/cbc:ID"/>
                                        <span class="AmountBoxHeader" data-id="{my:generateXPath($taxSchemeID)}">
                                            <xsl:value-of select="$taxSchemeID" />
                                        </span>
                                    </dd>
                                </xsl:if>

                            </xsl:for-each>

                        </dl>
                    </div>
                </xsl:for-each>
            </div>
        </xsl:for-each-group>
    </xsl:template>





    <!-- ========================================================================= -->
    <!-- Main Template Section
         Description: Defines the primary template for transforming the root Invoice or CreditNote into a complete HTML document,
                      including header, body sections, and embedded JavaScript for interactivity. -->
    <!-- ========================================================================= -->

    <xsl:template match="/cre:CreditNote">
        <html>
            <head>
                <meta charset="UTF-8" />
                <title>OIOUBL visning</title>


                <script type="text/javascript">
                    <![CDATA[
                    function toggleAllInfo(element) {
                        // Find nærmeste container der kan bruges til at finde toggle-info
                        let parentContainer = element.closest('.UBLAddRefLine') || element.closest('.UBLInfo');
                        
                        if (! parentContainer) {
                            console.error("Could not find parent UBLAddRefLine or UBLInfo container.");
                            return;
                        }
                        
                        let toggleInfo = null;
                        
                        // Hvis vi er i en tabelrække
                        if (parentContainer.tagName.toLowerCase() === 'tr') {
                            toggleInfo = parentContainer.nextElementSibling;
                        } else {
                            // Hvis strukturen er med en toggle-info direkte efter h3
                            // eller som søskende i .UBLAddRefLine / .UBLInfo
                            toggleInfo = element.nextElementSibling;
                            
                            // fallback hvis det ikke virker
                            if (! toggleInfo || ! toggleInfo.classList.contains('toggle-info')) {
                                toggleInfo = parentContainer.querySelector('.toggle-info');
                            }
                        }
                        
                        if (! toggleInfo || ! toggleInfo.classList.contains('toggle-info')) {
                            console.error("Could not find toggle-info element. Found:", toggleInfo);
                            return;
                        }
                        
                        let isTableRow = toggleInfo.tagName.toLowerCase() === 'tr';
                        let displayStyle = isTableRow ? "table-row": "block";
                        
                        if (toggleInfo.style.display === "none" || toggleInfo.style.display === "") {
                            toggleInfo.style.display = displayStyle;
                            element.classList.add("open");
                        } else {
                            toggleInfo.style.display = "none";
                            element.classList.remove("open");
                        }
                    }//]]>
                </script>

                <style type="text/css">
                    <xsl:value-of select="$css-content" disable-output-escaping="yes" />
                </style>
            </head>
            <body>
                <noscript>
                    <style type="text/css">
                        .toggle-info {
                            display: table-row !important;
                        }
                        .toggle-button::after {
                            content: "";
                        }</style>
                </noscript>

                <!-- <div class="print-button-container">
                    <button class="no-print" onclick="window.print()">Print dokument</button>
                </div>-->

                <header class="document-header">
                    <div class="header-container">

                        <!-- Tina -->
                        <!-- <div class="header-logo" /> -->
                        <div class="header-logo" id="logo-click" />
                        <!-- Tina -->
                        <h1 class="document-type">
                            <xsl:choose>
                                <xsl:when test="local-name() = 'Invoice'">FAKTURA</xsl:when>
                                <xsl:otherwise>KREDITNOTA</xsl:otherwise>
                            </xsl:choose>
                        </h1>

                        <div class="header-document-type-code credit-note-document-type-code">
                            <dt title="Kode der specificere typen af en kreditnota. Koden skal være fra OIOUBL UNCL 1001 Credit Note Subset">Kreditnota-type:</dt>
                            <xsl:variable name="CreditNoteTypeCode" select="cbc:CreditNoteTypeCode"/>
                            <dd data-id="{my:generateXPath($CreditNoteTypeCode)}">
                                <xsl:value-of select="cl:lookup('CreditNote/cbc:CreditNoteTypeCode', $CreditNoteTypeCode)" />
                            </dd>
                        </div>

                        <dl class="header-details">
                            <xsl:if test="cbc:ID">
                                <div class="header-row">
                                    <dt title="Løbenummer tildelt af afsender. Skal være dokumentets kreditnota-nummer til bogføring.">Kreditnota-nummer:</dt>
                                    <xsl:variable name="ID" select="cbc:ID"/>
                                    <dd data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="$ID" />
                                    </dd>
                                </div>
                            </xsl:if>
                            <xsl:if test="cbc:IssueDate">
                                <div class="header-row">
                                    <dt title="Datoen tildelt af Kreditor for hvornår kreditnotaen er udstedt. I Nemhandel forventes dokumenter at være sendt samme dag som de er udstedt.">Udstedelsesdato:</dt>
                                    <xsl:variable name="issueDate" select="cbc:IssueDate"/>
                                    <dd data-id="{my:generateXPath($issueDate)}">
                                        <xsl:value-of select="$issueDate" />
                                    </dd>
                                </div>
                            </xsl:if>
                            <xsl:if test="cac:OrderReference/cbc:ID">
                                <div class="header-row">
                                    <dt title="Købers ordrenummer. Hvis køber ikke har angivet et ordrenummer kan salgsordrenummeret benyttes i stedet.">Ordrenummer:</dt>
                                    <xsl:variable name="ID" select="cac:OrderReference/cbc:ID"/>
                                    <dd data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="$ID" />
                                    </dd>
                                </div>
                            </xsl:if>
                            <xsl:if test="cbc:DocumentCurrencyCode">
                                <div class="header-row">
                                    <dt title="Den valuta som kreditors og debitors mellemværende afregnes i.">Afregningsvaluta:</dt>
                                    <xsl:variable name="documentCurrencyCode" select="cbc:DocumentCurrencyCode"/>
                                    <dd data-id="{my:generateXPath($documentCurrencyCode)}">
                                        <xsl:value-of select="cl:lookup('CreditNote/cbc:DocumentCurrencyCode', $documentCurrencyCode)" />
                                    </dd>
                                </div>
                            </xsl:if>
                        </dl>
                    </div>
                </header>
                <hr class="header-separator" />

                <section class="parties">
                    <div class="party-container">
                        <xsl:apply-templates select="cac:AccountingCustomerParty" />
                    </div>
                    <div class="party-container">
                        <xsl:apply-templates select="cac:AccountingSupplierParty" />
                    </div>
                </section>


                <xsl:if test="cac:Delivery">
                    <section class="UBLInfo">
                        <xsl:call-template name="Levering" />
                    </section>
                </xsl:if>


                <xsl:if test="cbc:Note">
                    <section class="UBLInfo">
                        <xsl:call-template name="YderligInfo" />
                    </section>
                </xsl:if>

                <section class="items">
                    <table class="ItemsTable">
                        <thead>
                            <tr class="UBLLineHeader">
                                <th style="width: 5%;">Linje</th>
                                <th style="width: 10%;">Varenr</th>
                                <th style="width: 37%;">Beskrivelse</th>
                                <th style="width: 7%;">Antal</th>
                                <th style="width: 7%;">Enhed</th>
                                <th style="width: 10%;">Enhedspris</th>
                                <th style="width: 8%;">Moms</th>
                                <th style="width: 15%;">Pris</th>

                            </tr>
                        </thead>
                        <tbody>
                            <xsl:for-each select="cac:CreditNoteLine">
                                <xsl:variable name="rowClass" select="
                                        if (position() mod 2 = 1) then
                                            'oddrow'
                                        else
                                            'evenrow'" />
                                <tr class="UBLLine {$rowClass}">

                                    <xsl:variable name="ID" select="cbc:ID"/>
                                    <td class="UBLID" data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="$ID" />
                                    </td>

                                    <xsl:variable name="sellersItemIdentificationID" select="cac:Item/cac:SellersItemIdentification/cbc:ID"/>
                                    <td class="UBLSellersItemIdentification" data-id="{my:generateXPath($sellersItemIdentificationID)}">
                                        <xsl:value-of select="$sellersItemIdentificationID" />
                                    </td>

                                    <xsl:variable name="itemName" select="cac:Item/cbc:Name"/>
                                    <td class="UBLName" data-id="{my:generateXPath($itemName)}">
                                        <xsl:value-of select="$itemName" />
                                        <div class="UBLAddLineInfoWrapper">
                                            <xsl:call-template name="renderLineItemDetails" />
                                        </div>
                                    </td>

                                    <xsl:variable name="quantity" select="cbc:CreditedQuantity | cbc:InvoicedQuantity"/>
                                    <td class="UBLInvoicedQuantity" data-id="{my:generateXPath($quantity)}">
                                        <xsl:value-of select="$quantity" />
                                    </td>

                                    <xsl:variable name="unitCode" select="cbc:CreditedQuantity/@unitCode | cbc:InvoicedQuantity/@unitCode"/>
                                    <td class="UBLInvoiceQuantityUnit" data-id="{my:generateXPath($unitCode)}">
                                        <xsl:value-of select="cl:lookup('CreditNote/cac:CreditNoteLine/cbc:CreditedQuantity/@UnitCode', $unitCode)" />
                                    </td>


                                    <xsl:variable name="priceAmount" select="cac:Price/cbc:PriceAmount"/>
                                    <xsl:variable name="priceAmountCurrencyID" select="cac:Price/cbc:PriceAmount/@currencyID"/>
                                    <td class="UBLPriceUnit" data-id="{my:generateXPath($priceAmount)};{my:generateXPath($priceAmountCurrencyID)}">
                                        <span class="LineCurrencyID">
                                            <xsl:value-of select="cl:lookup('CreditNote/cac:CreditNoteLine/cac:Price/cbc:PriceAmount/@CurrencyID', $priceAmountCurrencyID)" />
                                        </span>
                                        <span class="AmountValue">
                                            <xsl:value-of select="format-number($priceAmount, '###,###,##0.00')"/>
                                        </span>
                                    </td>

                                    <xsl:variable name="percent" select="cac:TaxTotal/cac:TaxSubtotal/cac:TaxCategory/cbc:Percent"/>
                                    <xsl:variable name="taxCategoryID" select="cac:TaxTotal/cac:TaxSubtotal/cac:TaxCategory/cbc:ID"/>
                                    <td class="UBLTax" data-id="{my:generateXPath($percent)};{my:generateXPath($taxCategoryID)}">
                                        <xsl:value-of
                                            select="concat(format-number($percent, '#,##'), '% (', my:map-momsKategori($taxCategoryID), ')')"
                                         />
                                    </td>

                                    <xsl:variable name="lineExtensionAmount" select="cbc:LineExtensionAmount"/>
                                    <xsl:variable name="lineExtensionAmountCurrencyID" select="cbc:LineExtensionAmount/@currencyID"/>
                                    <td class="UBLLineExtensionAmount AlignRightTd" data-id="{my:generateXPath($lineExtensionAmount)};{my:generateXPath($lineExtensionAmountCurrencyID)}">
                                        <span class="LineCurrencyID">
                                            <xsl:value-of select="cl:lookup('CreditNote/cac:CreditNoteLine/cbc:LineExtensionAmount/@currencyID', $lineExtensionAmountCurrencyID)" />
                                        </span>
                                        <span class="AmountValue">
                                            <xsl:value-of select="format-number($lineExtensionAmount, '###,###,##0.00')"/>
                                        </span>
                                    </td>

                                </tr>
                                <tr class="empty-row">
                                    <td colspan="9" />
                                </tr>
                            </xsl:for-each>
                        </tbody>
                    </table>
                </section>

                <section class="totals">
                    <hr class="HrTaxTotalBorder" />
                    <table class="TaxTable">
                        <xsl:call-template name="renderTotals" />
                    </table>
                </section>


                <section>
                    <hr />
                    <div class="UBLInfo ubl-block">
                        <div class="section-divider">
                            <span class="toggle-button toggle-button" onclick="toggleAllInfo(this)">Moms angivet på hovedniveau</span>
                        </div>
                        <div class="toggle-info" style="display: none;">
                            <xsl:call-template name="render-tax-total" />
                        </div>
                    </div>
                </section>



                <xsl:if test="cac:AllowanceCharge">
                    <section>
                        <hr />
                        <div class="UBLInfo ubl-block">
                            <div class="section-divider">
                                <span class="toggle-button toggle-button" onclick="toggleAllInfo(this)">Rabat og afgifter på hovedniveau</span>
                            </div>
                            <div class="toggle-info" style="display: none;">
                                <xsl:call-template name="render-allowance-charge" />
                            </div>
                        </div>
                    </section>
                </xsl:if>



                <section class="payment">
                    <xsl:apply-templates select="cac:PaymentTerms" />
                    <xsl:apply-templates select="cac:PaymentMeans" />
                </section>


                <section class="document-params">
                    <xsl:call-template name="Dokumentparametre" />
                </section>
            </body>
        </html>
    </xsl:template>

    <!-- ========================================================================= -->
    <!-- Line Item Details Template Section
         Description: Handles the rendering of detailed line item information, including descriptions, order references,
                      additional properties, certificates, and allowances/charges with collapsible sections. -->
    <!-- ========================================================================= -->

    <xsl:template name="renderLineItemDetails">
        <table class="UBLAddLineInfo">
            <xsl:if test="cac:Item/cbc:Description">
                <tr>
                    <td title="Et fritekstfelt, der kan bruges til at angive en tekstbeskrivelse af varen." class="UBLAddLineInfo-label">Beskrivelse:</td>

                    <xsl:variable name="description" select="cac:Item/cbc:Description"/>
                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($description)}">
                        <xsl:value-of select="$description" />
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="cbc:Note">
                <tr>
                    <td title="Fritekst vedrørende kreditnota-linjen. Dette element kan eventuelt indeholde noter eller lignende oplysninger, der ikke udtrykkeligt fremgår i andre strukturer." class="UBLAddLineInfo-label">Note:</td>

                    <xsl:variable name="note" select="cbc:Note"/>
                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($note)}">
                        <xsl:value-of select="$note" />
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="cac:Item/cac:SellersItemIdentification/cbc:ID">
                <tr>
                    <td title="Varenummer tildelt af sælger" class="UBLAddLineInfo-label">Sælgers varenummer.:</td>

                    <xsl:variable name="ID" select="cac:Item/cac:SellersItemIdentification/cbc:ID"/>
                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($ID)}">
                        <xsl:value-of select="$ID" />
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="cac:OrderLineReference/cbc:LineID or cac:OrderLineReference/cac:OrderReference/cbc:ID or cac:OrderLineReference/cac:OrderReference/cbc:IssueDate">
                <tr class="UBLAddRefLine">
                    <td colspan="2">
                        <div class="section-divider">
                            <span class="toggle-button" onclick="toggleAllInfo(this)">Orderinfo</span>
                        </div>
                    </td>
                </tr>
                <tr class="toggle-info" style="display: none;">
                    <td colspan="2">
                        <table class="UBLAddLineInfo">
                            <xsl:if test="cac:OrderLineReference/cbc:LineID">
                                <tr>
                                    <td title="Identifikationen af den relevante linje i den bestilling der henvises til." class="UBLAddLineInfo-label">Ordrelinjenummer.:</td>
                                    <td class="UBLAddLineInfo-value">
                                        <xsl:variable name="lineID" select="cac:OrderLineReference/cbc:LineID"/>
                                        <span class="AmountBox" data-id="{my:generateXPath($lineID)}">
                                            <xsl:value-of select="$lineID" />
                                        </span>
                                    </td>
                                </tr>
                            </xsl:if>
                            <xsl:if test="cac:OrderLineReference/cac:OrderReference/cbc:ID">
                                <tr>
                                    <td title="Bestillingsreferencen for den bestilling der refereres til." class="UBLAddLineInfo-label">Bestillingsnummer:</td>

                                    <xsl:variable name="ID" select="cac:OrderLineReference/cac:OrderReference/cbc:ID"/>
                                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="$ID" />
                                    </td>
                                </tr>
                            </xsl:if>
                            <xsl:if test="cac:OrderLineReference/cac:OrderReference/cbc:IssueDate">
                                <tr>
                                    <td title="Dato for udstedelse af den bestilling, der refereres til." class="UBLAddLineInfo-label">Udstedelsesdato:</td>

                                    <xsl:variable name="issueDate" select="cac:OrderLineReference/cac:OrderReference/cbc:IssueDate"/>
                                    <td class="UBLAddLineInfo-value" data-id="{$issueDate}">
                                        <xsl:value-of select="$issueDate" />
                                    </td>
                                </tr>
                            </xsl:if>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="cac:Item/cac:AdditionalItemProperty">
                <tr class="UBLAddRefLine">
                    <td colspan="2">
                        <div class="section-divider">
                            <span  class="toggle-button" onclick="toggleAllInfo(this)">Supplerende Oplysninger</span>
                        </div>
                    </td>
                </tr>
                <tr class="toggle-info" style="display: none;">
                    <td colspan="2">
                        <table class="UBLAddLineInfo">
                            <xsl:for-each select="cac:Item/cac:AdditionalItemProperty">
                                <tr>
                                    <td class="UBLAddLineInfo-label">Name/Value:</td>

                                    <xsl:variable name="name" select="cbc:Name"/>
                                    <xsl:variable name="value" select="cbc:Value"/>
                                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($name)};{my:generateXPath($value)}">
                                        <xsl:value-of select="concat($name, ' / ', $value)" />
                                    </td>
                                </tr>
                            </xsl:for-each>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="cac:Item/cac:Certificate">
                <tr class="UBLAddRefLine">
                    <td colspan="2">
                        <div class="section-divider">
                            <span class="toggle-button" onclick="toggleAllInfo(this)">Certificate</span>
                        </div>
                    </td>
                </tr>
                <tr class="toggle-info" style="display: none;">
                    <td colspan="2">
                        <table class="UBLAddLineInfo">
                            <xsl:if test="cac:Item/cac:Certificate/cbc:ID">
                                <tr>
                                    <td title="Certifikatet eller mærkets nummer - kan være et ID-nummer udstedt af certificeringsautoriteten, eller blot et løbenummer til intern sporing i dokumentet." class="UBLAddLineInfo-label">Certifikatnummer:</td>

                                    <xsl:variable name="ID" select="cac:Item/cac:Certificate/cbc:ID"/>
                                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="cl:lookup('CreditNote/cac:CreditNoteLine/cac:Item/cac:Certificate/cbc:ID', $ID)" />
                                    </td>
                                </tr>
                            </xsl:if>
                            <xsl:if test="cac:Item/cac:Certificate/cbc:Remarks">
                                <tr>
                                    <td title="Fritekstfelt til yderligere oplysninger om mærkningen." class="UBLAddLineInfo-label">Kommentar:</td>

                                    <xsl:variable name="remarks" select="cac:Item/cac:Certificate/cbc:Remarks"/>
                                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($remarks)}">
                                        <xsl:value-of select="$remarks" />
                                    </td>
                                </tr>
                            </xsl:if>
                        </table>
                    </td>
                </tr>
            </xsl:if>


            <xsl:if test="cac:AllowanceCharge">
                <tr class="UBLAddRefLine">
                    <td colspan="2">
                        <div class="section-divider">
                            <span class="toggle-button" onclick="toggleAllInfo(this)">Rabat/Gebyr for linjen</span>
                        </div>
                    </td>
                </tr>
                <tr class="toggle-info" style="display: none;">
                    <td colspan="2">
                        <table class="UBLAddLineInfo">
                            <xsl:for-each select="cac:AllowanceCharge">


                                <xsl:if test="cbc:ChargeIndicator">
                                    <tr>
                                        <xsl:variable name="chargeIndicator" select="cbc:ChargeIndicator"/>
                                        <td class="UBLAddLineInfo-label" data-id="{my:generateXPath($chargeIndicator)}">
                                            <xsl:value-of select="concat(my:map-gebyrRabat($chargeIndicator), ':')" />
                                        </td>
                                        <td class="UBLAddLineInfo-value">
                                            <xsl:variable name="amount" select="cbc:Amount"/>
                                            <xsl:variable name="currencyID" select="cbc:Amount/@currencyID"/>
                                            <span class="AmountBox" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                                                <xsl:value-of select="concat(format-number(($amount), '###,###,##0.00'), ' ', cl:lookup('CreditNote/cac:CreditNoteLine/cac:AllowanceCharge/cbc:Amount/@currencyID', $currencyID))" />
                                            </span>
                                        </td>
                                    </tr>
                                </xsl:if>

                            </xsl:for-each>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="cac:Price/cac:AllowanceCharge">
                <tr class="UBLAddRefLine">
                    <td colspan="2">
                        <div class="section-divider">
                            <span class="toggle-button" onclick="toggleAllInfo(this)">Rabat/Gebyr for pris</span>
                        </div>
                    </td>
                </tr>
                <tr class="toggle-info" style="display: none;">
                    <td colspan="2">
                        <table class="UBLAddLineInfo">
                            <xsl:for-each select="cac:Price/cac:AllowanceCharge">

                                <xsl:if test="cbc:ChargeIndicator">
                                    <tr>
                                        <xsl:variable name="chargeIndicator" select="cbc:ChargeIndicator"/>
                                        <td class="UBLAddLineInfo-label" data-id="{my:generateXPath($chargeIndicator)}">
                                            <xsl:value-of select="concat(my:map-gebyrRabat($chargeIndicator), ':')" />
                                        </td>
                                        <td class="UBLAddLineInfo-value">
                                            <xsl:variable name="amount" select="cbc:Amount"/>
                                            <xsl:variable name="currencyID" select="cbc:Amount/@currencyID"/>
                                            <span class="AmountBox" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                                                <xsl:value-of select="concat(format-number(($amount), '###,###,##0.00'), ' ',  cl:lookup('CreditNote/cac:CreditNoteLine/cac:Price/cac:AllowanceCharge/cbc:Amount/@currencyID', $currencyID))" />
                                            </span>
                                        </td>
                                    </tr>
                                </xsl:if>
                            </xsl:for-each>
                        </table>
                    </td>
                </tr>
            </xsl:if>


            <xsl:if test="cac:TaxTotal">
                <tr class="UBLAddRefLine">
                    <td colspan="2">
                        <div class="section-divider">
                            <span class="toggle-button" onclick="toggleAllInfo(this)">Moms</span>
                        </div>
                    </td>
                </tr>
                <tr class="toggle-info" style="display: none;">
                    <td colspan="2">
                        <table class="UBLAddLineInfo">
                            <xsl:for-each select="cac:TaxTotal/cac:TaxSubtotal">

                                <tr>
                                    <td title="Momskoden for momskategorien." class="UBLAddLineInfo-label">Momskode:</td>
                                    <td class="UBLAddLineInfo-value">

                                        <xsl:variable name="taxCategoryID" select="cac:TaxCategory/cbc:ID"/>
                                        <span class="AmountBox" data-id="{my:generateXPath($taxCategoryID)}">
                                            <xsl:value-of select="concat($taxCategoryID, ' (', my:map-taxCategory($taxCategoryID), ')')" />
                                        </span>
                                    </td>

                                </tr>
                                <tr>
                                    <td title="Den samlede moms-subtotal." class="UBLAddLineInfo-label">Momsbeløb:</td>
                                    <td class="UBLAddLineInfo-value">
                                        <xsl:variable name="taxAmount" select="cbc:TaxAmount"/>
                                        <xsl:variable name="currencyID" select="cbc:TaxAmount/@currencyID"/>
                                        <span class="AmountBox" data-id="{my:generateXPath($taxAmount)};{my:generateXPath($currencyID)}">
                                            <xsl:value-of select="concat(format-number(($taxAmount), '###,###,##0.00'), ' ', cl:lookup('CreditNote/cac:CreditNoteLine/cac:TaxTotal/cac:TaxSubtotal/cbc:TaxAmount/@currencyID', $currencyID))" />
                                        </span>
                                    </td>
                                </tr>

                                <xsl:if test="cac:TaxCategory/cbc:TaxExemptionReasonCode">
                                    <tr>
                                        <td title="Grunden til momsfritagelse, angivet som kode." class=" UBLAddLineInfo-label">Momsfritagelseskode:</td>

                                        <xsl:variable name="taxExemptionReasonCode" select="cac:TaxCategory/cbc:TaxExemptionReasonCode"/>
                                        <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($taxExemptionReasonCode)}">
                                            <xsl:value-of select="$taxExemptionReasonCode" />
                                        </td>
                                    </tr>
                                </xsl:if>

                                <xsl:if test="cac:TaxCategory/cbc:TaxExemptionReason">
                                    <tr>
                                        <td title="Grunden til momsfritagelse." class="UBLAddLineInfo-label">Momsfritagelsesgrund:</td>
                                        <xsl:variable name="taxExemptionReason" select="cac:TaxCategory/cbc:TaxExemptionReason"/>
                                        <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($taxExemptionReason)}">
                                            <xsl:value-of select="$taxExemptionReason" />
                                        </td>
                                    </tr>
                                </xsl:if>

                            </xsl:for-each>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="cac:Item/cac:ClassifiedTaxCategory">
                <tr class="UBLAddRefLine">
                    <td colspan="2">
                        <div class="section-divider">
                            <span class="toggle-button" onclick="toggleAllInfo(this)">ItemMoms</span>
                        </div>
                    </td>
                </tr>
                <tr class="toggle-info" style="display: none;">
                    <td colspan="2">
                        <table class="UBLAddLineInfo">

                            <tr>
                                <td title="Momskoden for momskategorien." class="UBLAddLineInfo-label">Momskode:</td>
                                <td class="UBLAddLineInfo-value">

                                    <xsl:variable name="ID" select="cac:Item/cac:ClassifiedTaxCategory/cbc:ID"/>
                                    <span class="AmountBox" data-id="{my:generateXPath($ID)}">
                                        <xsl:value-of select="concat($ID, ' (', my:map-taxCategory($ID), ')')" />
                                    </span>
                                </td>

                            </tr>


                            <xsl:if test="cac:Item/cac:ClassifiedTaxCategory/cbc:Name">
                                <tr>
                                    <td title="Navnet på momskategorien." class="UBLAddLineInfo-label">Momskategori:</td>

                                    <xsl:variable name="name" select="cac:Item/cac:ClassifiedTaxCategory/cbc:Name"/>
                                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($name)}">
                                        <xsl:value-of select="$name" />
                                    </td>
                                </tr>
                            </xsl:if>


                            <xsl:if test="cac:Item/cac:ClassifiedTaxCategory/cbc:Percent">
                                <tr>
                                    <td title="Momskategoriens moms-sats, i procent." class="UBLAddLineInfo-label">Momssats:</td>
                                    <td class="UBLAddLineInfo-value">

                                        <xsl:variable name="percent" select="cac:Item/cac:ClassifiedTaxCategory/cbc:Percent"/>
                                        <span class="AmountBox" data-id="{my:generateXPath($percent)}">
                                            <xsl:value-of select="concat($percent, ' %')" />

                                        </span>
                                    </td>
                                </tr>
                            </xsl:if>

                            <xsl:if test="cac:Item/cac:ClassifiedTaxCategory/cbc:TaxExemptionReasonCode">
                                <tr>
                                    <td title="Grunden til momsfritagelse, angivet som kode." class="UBLAddLineInfo-label">Momsfritagelseskode:</td>

                                    <xsl:variable name="reasonCode" select="cac:Item/cac:ClassifiedTaxCategory/cbc:TaxExemptionReasonCode"/>
                                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($reasonCode)}">
                                        <xsl:value-of select="$reasonCode" />
                                    </td>
                                </tr>
                            </xsl:if>

                            <xsl:if test="cac:Item/cac:ClassifiedTaxCategory/cbc:TaxExemptionReason">
                                <tr>
                                    <td title="Grunden til momsfritagelse." class="UBLAddLineInfo-label">Momsfritagelsesgrund:</td>

                                    <xsl:variable name="reason" select="cac:Item/cac:ClassifiedTaxCategory/cbc:TaxExemptionReason"/>
                                    <td class="UBLAddLineInfo-value" data-id="{my:generateXPath($reason)}">
                                        <xsl:value-of select="$reason" />
                                    </td>
                                </tr>
                            </xsl:if>
                        </table>
                    </td>
                </tr>
            </xsl:if>
        </table>
    </xsl:template>

    <!-- ========================================================================= -->
    <!-- Totals Template Section
         Description: Renders the totals section of the document, including line sums, net sums, gross sums, and payable amounts,
                      with proper formatting for currency and decimal places. -->
    <!-- ========================================================================= -->

    <xsl:template name="renderTotals">
        <tr class="TaxTotal">
            <td colspan="2" class="TaxAndTotalText" title="Summen af linjetotalen for alle fakturalinjer, eksklusive moms og afgifter, men inklusive gebyrer og afrunding.">Linjesum</td>

            <xsl:variable name="lineExtensionAmount" select="cac:LegalMonetaryTotal/cbc:LineExtensionAmount"/>
            <xsl:variable name="currencyID" select="cac:LegalMonetaryTotal/cbc:LineExtensionAmount/@currencyID"/>
            <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($currencyID)}">
                <xsl:value-of select="cl:lookup('CreditNote/cac:LegalMonetaryTotal/cbc:LineExtensionAmount/@currencyID', $currencyID)" />
            </td>
            <td class="TaxAndTotalValue" data-id="{my:generateXPath($lineExtensionAmount)}">
                <xsl:value-of select="format-number(($lineExtensionAmount), '###,###,##0.00')" />
            </td>
        </tr>

        <tr class="TaxTotal">
            <td colspan="2" class="TaxAndTotalText" title="Summen af alle rabatter på dokumentniveau (men ikke linje-rabatter).">Rabatsum</td>

            <xsl:variable name="amount" select="(cac:LegalMonetaryTotal/cbc:AllowanceTotalAmount, 0)[1]" />
            <xsl:variable name="currencyID" select="(cac:LegalMonetaryTotal/cbc:AllowanceTotalAmount/@currencyID, cbc:DocumentCurrencyCode)[1]" />
            <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($currencyID)}">
                <xsl:value-of select="cl:lookup('CreditNote/cac:LegalMonetaryTotal/cbc:AllowanceTotalAmount/@currencyID', $currencyID)" />
            </td>
            <td class="TaxAndTotalValue" data-id="{my:generateXPath($amount)}">
                <xsl:value-of select="format-number(number($amount), '###,###,##0.00')" />
            </td>
        </tr>
        <tr class="TaxTotal">
            <td colspan="2" class="TaxAndTotalText" title="Summen af alle gebyrer på dokumentniveau (men ikke linje-rabatter).">Gebyrsum</td>

            <xsl:variable name="amount" select="(cac:LegalMonetaryTotal/cbc:ChargeTotalAmount, 0)[1]" />
            <xsl:variable name="currencyID" select="(cac:LegalMonetaryTotal/cbc:ChargeTotalAmount/@currencyID, cbc:DocumentCurrencyCode)[1]" />
            <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="cl:lookup('CreditNote/cac:LegalMonetaryTotal/cbc:ChargeTotalAmount/@currencyID', $currencyID)" />
            </td>
            <td class="TaxAndTotalValue" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="format-number(($amount), '###,###,##0.00')" />
            </td>
        </tr>

        <tr class="TaxTotalBorder">
            <td colspan="4" style="border-top: 2px solid black;" />
        </tr>

        <tr class="TaxTotal">
            <td colspan="2" class="TaxAndTotalText" title="Den samlede fakturasum før moms, men inklusive afgifter, rabatter og gebyrer udover rabatter og gebyrer relateret til betalingsbetingelserne.">Nettosum</td>

            <xsl:variable name="taxExclusiveAmount" select="cac:LegalMonetaryTotal/cbc:TaxExclusiveAmount"/>
            <xsl:variable name="currencyID" select="cac:LegalMonetaryTotal/cbc:TaxExclusiveAmount/@currencyID"/>
            <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($taxExclusiveAmount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="cl:lookup('CreditNote/cac:LegalMonetaryTotal/cbc:TaxExclusiveAmount/@currencyID', $currencyID)" />
            </td>
            <td class="TaxAndTotalValue" data-id="{my:generateXPath($taxExclusiveAmount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="format-number(($taxExclusiveAmount), '###,###,##0.00')" />
            </td>
        </tr>
        <tr class="TaxTotal">
            <td colspan="2" class="TaxAndTotalText" title="Den samlede moms og skat for gebyret for denne moms-kategori.">Moms-sum</td>

            <xsl:variable name="taxAmount" select="(cac:TaxTotal/cbc:TaxAmount, 0)[1]" />
            <xsl:variable name="currencyID" select="(cac:TaxTotal/cbc:TaxAmount/@currencyID, cbc:DocumentCurrencyCode)[1]" />
            <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($taxAmount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="cl:lookup('CreditNote/cac:TaxTotal/cbc:TaxAmount/@currencyID', $currencyID)" />
            </td>
            <td class="TaxAndTotalValue" data-id="{my:generateXPath($taxAmount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="format-number(($taxAmount), '###,###,##0.00')" />
            </td>
        </tr>

        <tr class="TaxTotalBorder">
            <td colspan="4" style="border-top: 2px solid black;" />
        </tr>

        <tr class="TaxTotal">
            <td colspan="2" class="TaxAndTotalText" title="Den samlede fakturasum inklusive moms.">Bruttosum</td>

            <xsl:variable name="taxInclusiveAmount" select="cac:LegalMonetaryTotal/cbc:TaxInclusiveAmount"/>
            <xsl:variable name="currencyID" select="cac:LegalMonetaryTotal/cbc:TaxInclusiveAmount/@currencyID"/>
            <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($taxInclusiveAmount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="cl:lookup('CreditNote/cac:LegalMonetaryTotal/cbc:TaxInclusiveAmount/@currencyID', $currencyID)" />
            </td>
            <td class="TaxAndTotalValue" data-id="{my:generateXPath($taxInclusiveAmount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="format-number(($taxInclusiveAmount), '###,###,##0.00')" />
            </td>
        </tr>
        <tr class="TaxTotal">
            <td colspan="2" class="TaxAndTotalText" title="Det samlede beløb som allerede er betalt.">Forudbetaling</td>

            <xsl:variable name="amount" select="(cac:LegalMonetaryTotal/cbc:PrepaidAmount, 0)[1]" />
            <xsl:variable name="currencyID" select="(cac:LegalMonetaryTotal/cbc:PrepaidAmount/@currencyID, cbc:DocumentCurrencyCode)[1]" />
            <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="cl:lookup('CreditNote/cac:LegalMonetaryTotal/cbc:PrepaidAmount/@currencyID', $currencyID)" />
            </td>
            <td class="TaxAndTotalValue" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="format-number(($amount), '###,###,##0.00')" />
            </td>
        </tr>
        <tr class="TaxTotal">
            <td colspan="2" class="TaxAndTotalText" title="Afrundingsbeløb, der sikrer at PayableAmount = TaxInclusiveAmount + PayableRoundingAmount">Afrunding</td>

            <xsl:variable name="amount" select="(cac:LegalMonetaryTotal/cbc:PayableRoundingAmount, 0)[1]" />
            <xsl:variable name="currencyID" select="(cac:LegalMonetaryTotal/cbc:PayableRoundingAmount/@currencyID, cbc:DocumentCurrencyCode)[1]" />
            <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="cl:lookup('CreditNote/cac:LegalMonetaryTotal/cbc:PayableRoundingAmount/@currencyID', $currencyID)" />
            </td>
            <td class="TaxAndTotalValue" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                <xsl:value-of select="format-number(($amount), '###,###,##0.00')" />
            </td>
        </tr>
        <tr class="TaxTotalBorder">
            <td colspan="4" style="border-top: 2px solid black;" />
        </tr>
        <tr class="TaxTotal TaxTotalHighlighted">
            <td colspan="2" class="TaxAndTotalText" title="Det samlede beløb til betaling: TaxInclusiveAmount - PrepaidAmount.">Beløb til betaling</td>

            <xsl:variable name="amount" select="cac:LegalMonetaryTotal/cbc:PayableAmount"/>
            <xsl:variable name="currencyID" select="cac:LegalMonetaryTotal/cbc:PayableAmount/@currencyID"/>
            <td class="TaxAndTotalCurrency" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                <b>
                    <xsl:value-of select="cl:lookup('CreditNote/cac:LegalMonetaryTotal/cbc:PayableAmount/@currencyID', $currencyID)" />
                </b>
            </td>
            <td class="TaxAndTotalValue" data-id="{my:generateXPath($amount)};{my:generateXPath($currencyID)}">
                <b>
                    <xsl:value-of select="format-number(($amount), '###,###,##0.00')" />
                </b>
            </td>
        </tr>
    </xsl:template>

</xsl:stylesheet>
